/*:
 * @target MZ
 * @plugindesc RSTH_StatusHud:ステータスHUD表示プラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 * 
 * @param ThirstVariableId
 * @text 渇き度の変数ID
 * @type variable
 * @default 1
 *
 * @param HungerVariableId
 * @text 空腹度の変数ID
 * @type variable
 * @default 2
 *
 */


(() => {
    "use strict";

    window.RSTH_IH = window.RSTH_IH || {};

    const p = PluginManager.parameters("RSTH_StatusHud") || {};
    const thirstVarId = Number(p["ThirstVariableId"] || 1);
    const hungerVarId = Number(p["HungerVariableId"] || 2);



    window.RSTH_IH.MiniMapSprite = class extends Sprite {
        constructor() {
            const size = 160;
            const bitmap = new Bitmap(size, size);
            super(bitmap);
            this._size = size;
            this.x = Graphics.width - size - 10;
            this.y = 10;
            this.opacity = 230;
            this._lastPlayerX = -1;
            this._lastPlayerY = -1;
            this._lastMobPositions = "";
            this._backgroundBitmap = null;
            this._scaleX = 1;
            this._scaleY = 1;

            this.createBackground();

            // ✅ MobCount背景＋テキスト
            this._mobCountBg = new Sprite(new Bitmap(160, 24));
            this._mobCountBg.x = -170;
            this._mobCountBg.y = 0;
            this._mobCountBg.bitmap.fillAll("#000000");
            this._mobCountBg.alpha = 0.6;
            this.addChild(this._mobCountBg);

            this._mobCountText = new Sprite(new Bitmap(160, 24));
            this._mobCountText.x = this._mobCountBg.x;
            this._mobCountText.y = this._mobCountBg.y;
            this._mobCountText.bitmap.fontSize = 18;
            this._mobCountText.bitmap.textColor = "#ffffff";
            this._mobCountText.bitmap.outlineColor = "#000000";
            this.addChild(this._mobCountText);

            // ✅ MobDCount背景＋テキスト
            this._mobDCountBg = new Sprite(new Bitmap(160, 72));
            this._mobDCountBg.x = -170;
            this._mobDCountBg.y = 28;
            this._mobDCountBg.bitmap.fillAll("#000000");
            this._mobDCountBg.alpha = 0.6;
            this.addChild(this._mobDCountBg);

            this._mobDCountText = new Sprite(new Bitmap(160, 72));
            this._mobDCountText.x = this._mobDCountBg.x;
            this._mobDCountText.y = this._mobDCountBg.y;
            this._mobDCountText.bitmap.fontSize = 18;
            this._mobDCountText.bitmap.textColor = "#ffffff";
            this._mobDCountText.bitmap.outlineColor = "#000000";
            this.addChild(this._mobDCountText);

            // ✅ 所持金背景＋テキスト
            this._goldBg = new Sprite(new Bitmap(160, 48));
            this._goldBg.x = -170;
            this._goldBg.y = 104; // MobDCountの下に配置
            this._goldBg.bitmap.fillAll("#000000");
            this._goldBg.alpha = 0.6;
            this.addChild(this._goldBg);

            this._goldText = new Sprite(new Bitmap(160, 48));
            this._goldText.x = this._goldBg.x;
            this._goldText.y = this._goldBg.y;
            this._goldText.bitmap.fontSize = 18;
            this._goldText.bitmap.textColor = "#ffffff";
            this._goldText.bitmap.outlineColor = "#000000";
            this.addChild(this._goldText);

            this._lastMobCount = -1;
            this._lastMobDCount = -1;
            this._lastGold = -1;

            this.updateMobCountText();
            this.updateMobDCountText();
            this.updateGoldText();
        }

        createBackground() {
            const mapWidth = $gameMap.width();
            const mapHeight = $gameMap.height();
            const size = this._size;
            this._scaleX = size / mapWidth;
            this._scaleY = size / mapHeight;

            const bg = new Bitmap(size, size);
            const ctx = bg.context;

            for (let y = 0; y < mapHeight; y++) {
                for (let x = 0; x < mapWidth; x++) {
                    const pass = $gameMap.isPassable(x, y, 2);
                    ctx.fillStyle = pass ? "#888888" : "#333333";
                    ctx.fillRect(x * this._scaleX, y * this._scaleY, this._scaleX, this._scaleY);
                }
            }

            bg._setDirty?.();
            this._backgroundBitmap = bg;
        }

        update() {
            super.update();

            const playerX = $gamePlayer.x;
            const playerY = $gamePlayer.y;

            const mobPositions = Object.values(window.RSTH_IH._mobManager?._mobs || {})
                .map(m => `${m._x},${m._y}`)
                .join("|");

            if (
                playerX !== this._lastPlayerX ||
                playerY !== this._lastPlayerY ||
                mobPositions !== this._lastMobPositions
            ) {
                this.drawMiniMap();
                this._lastPlayerX = playerX;
                this._lastPlayerY = playerY;
                this._lastMobPositions = mobPositions;
            }

            this.updateMobCountText();
            this.updateMobDCountText();
            this.updateGoldText();
        }

        updateMobCountText() {
            const mobManager = window.RSTH_IH._mobManager;
            const mobCount = mobManager ? Object.keys(mobManager._mobs).length : 0;
            if (mobCount !== this._lastMobCount) {
                this._lastMobCount = mobCount;
                const text = `総MOB数：${mobCount}体`;

                const bgBmp = this._mobCountBg.bitmap;
                const textBmp = this._mobCountText.bitmap;

                // 背景色を変更
                bgBmp.clear();
                if (mobCount > 180) {
                    bgBmp.fillAll("#ff0000"); // 赤
                } else if (mobCount > 120) {
                    bgBmp.fillAll("#ffff00"); // 黄
                } else {
                    bgBmp.fillAll("#000000"); // 通常
                }

                // テキスト再描画
                textBmp.clear();
                textBmp.drawText(text, 0, 0, textBmp.width, 24, 'center');
            }
        }


        updateMobDCountText() {
            const mobDCount = window.RSTH_IH.TreasureMobCount - $gameSystem._defeatedTreasureMobCount;
            if (mobDCount !== this._lastMobDCount) {
                this._lastMobDCount = mobDCount;

                const text1 = `宝箱出現まで`;
                const text2 = `必要な討伐数：`;
                const text3 = `${mobDCount}体`;

                const bmp = this._mobDCountText.bitmap;
                bmp.clear();
                bmp.drawText(text1, 0, 0, bmp.width, 24, 'center');
                bmp.drawText(text2, 0, 24, bmp.width, 24, 'center');
                bmp.drawText(text3, 0, 48, bmp.width, 24, 'center');
            }
        }

        updateGoldText() {
            const gold = $gameParty.gold();
            if (gold !== this._lastGold) {
                this._lastGold = gold;

                const bmp = this._goldText.bitmap;
                bmp.clear();
                bmp.drawText(`所持金`, 0, 0, bmp.width, 24, 'center');
                bmp.drawText(`${gold} zeny`, 0, 24, bmp.width, 24, 'center');
            }
        }

        drawMiniMap() {
            const bitmap = this.bitmap;
            const ctx = bitmap.context;

            bitmap.clear();

            if (this._backgroundBitmap) {
                bitmap.blt(this._backgroundBitmap, 0, 0, this._size, this._size, 0, 0);
            }

            for (const mob of Object.values(window.RSTH_IH._mobManager?._mobs || {})) {
                const mx = mob._x * this._scaleX;
                const my = mob._y * this._scaleY;
                ctx.fillStyle = "#ff0000";
                ctx.fillRect(mx, my, this._scaleX, this._scaleY);
            }

            const px = $gamePlayer.x * this._scaleX;
            const py = $gamePlayer.y * this._scaleY;
            ctx.fillStyle = "#00ff00";
            ctx.fillRect(px, py, this._scaleX, this._scaleY);
        }
    };



    // ステータスHUDクラス
    class Window_StatusHUD extends Window_Base {
        constructor(rect) {
            super(rect);
            this.setBackgroundType(0); // 通常ウィンドウ（枠あり）
            this.opacity = 255; // 枠の不透明度
            this.contents.paintOpacity = 128; // 中身は半透明
            this._lastHp = -1;
            this._lastMp = -1;
            this._faceBitmap = null;
            this.loadFaceBitmap();
        }

        loadFaceBitmap() {
            const actor = $gameParty.leader();
            if (!actor) return;

            const bitmap = ImageManager.loadFace(actor.faceName());
            bitmap.addLoadListener(() => {
                this._faceBitmap = bitmap;
                this._faceIndex = actor.faceIndex();
                this.refresh();
            });
        }

        refresh() {
            this.contents.clear();
            const actor = $gameParty.leader();
            if (!actor || !this._faceBitmap) return;

            const faceW = 144, faceH = 144, cols = 4, rows = 2;
            const index = actor.faceIndex();
            const sx = (index % cols) * faceW;
            const sy = Math.floor(index / cols) * faceH;
            const destSize = 96, dx = 6, dy = 6;

            const ctx = this.contents.context;

            // 背景丸
            ctx.save();
            ctx.globalAlpha = 0.6;
            ctx.beginPath();
            ctx.arc(dx + destSize / 2, dy + destSize / 2, destSize / 2, 0, Math.PI * 2);
            ctx.fillStyle = "#000000";
            ctx.fill();
            ctx.restore();

            // 顔円マスク
            ctx.save();
            ctx.beginPath();
            ctx.arc(dx + destSize / 2, dy + destSize / 2, destSize / 2, 0, Math.PI * 2);
            ctx.clip();
            this.contents.blt(this._faceBitmap, sx, sy, faceW, faceH, dx, dy, destSize, destSize);
            ctx.restore();

            // 顔の縁
            ctx.save();
            ctx.beginPath();
            ctx.arc(dx + destSize / 2, dy + destSize / 2, destSize / 2, 0, Math.PI * 2);
            ctx.strokeStyle = "rgba(25, 26, 44, 1)";
            ctx.lineWidth = 5;
            ctx.stroke();
            ctx.restore();

            // Lv表示（顔の左下あたり）
            const lvText = `Lv.${actor.level}`;

            // 🔽 フォント状態を退避
            const prevFontSize = this.contents.fontSize;
            const prevTextColor = this.contents.textColor;
            const prevOutlineColor = this.contents.outlineColor;
            const prevOutlineWidth = this.contents.outlineWidth;

            // 🔽 小さなフォントで描画
            this.contents.fontSize = 22;
            this.contents.textColor = "#ffffff";
            this.contents.outlineColor = "#000000";
            this.contents.outlineWidth = 4;
            this.drawText(lvText, dx + 4, dy + destSize - 24, destSize - 8, 'left');

            // 🔽 フォント状態を復元
            this.contents.fontSize = prevFontSize;
            this.contents.textColor = prevTextColor;
            this.contents.outlineColor = prevOutlineColor;
            this.contents.outlineWidth = prevOutlineWidth;


            // 名前・HP・MP
            const nameX = dx + destSize + 8;
            const nameY = dy;
            this.contents.paintOpacity = 255;
            this.drawText(actor.name(), nameX, nameY, 196, 'left');

            const hpY = nameY + 32;
            this.drawGaugeBar(nameX, hpY, 196, actor.hpRate(), ColorManager.hpGaugeColor1(), ColorManager.hpGaugeColor2());
            this.drawText(`HP: ${actor.hp} / ${actor.mhp}`, nameX, hpY, 196, 'left');

            const mpY = hpY + 32;
            this.drawGaugeBar(nameX, mpY, 196, actor.mpRate(), ColorManager.mpGaugeColor1(), ColorManager.mpGaugeColor2());
            this.drawText(`SP: ${actor.mp} / ${actor.mmp}`, nameX, mpY, 196, 'left');
        }

        update() {
            super.update();
            const actor = $gameParty.leader();
            //console.log(actor);
            if (!actor) return;

            if (this._lastHp !== actor.hp ||
                this._lastMp !== actor.mp
            ) { // 🔽 追加条件
                this._lastHp = actor.hp;
                this._lastMp = actor.mp;
                this.refresh();
            }
        }

        drawGaugeBar(x, y, width, rate, color1, color2) {
            const fillW = Math.floor(width * rate);
            const gaugeH = 12;
            const gaugeY = y + 20;
            this.contents.fillRect(x, gaugeY, width, gaugeH, ColorManager.gaugeBackColor());
            this.contents.gradientFillRect(x, gaugeY, fillW, gaugeH, color1, color2);
        }
    }

    class Window_StatusSubHUD extends Window_Base {
        constructor(rect) {
            super(rect);
            this.setBackgroundType(0);
            this.opacity = 255;
            this.contents.paintOpacity = 255;
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            const actor = $gameParty.leader();
            if (!actor) return;

            let atk = actor.param(2);
            if (actor.hasState(14)) atk = Math.round(atk * 1.25);

            let matk = actor.param(4);
            let def = actor.param(3);
            let mdef = Math.round(actor.param(4) / 2);
            let slot = null;
            let item = null;

            let hindex = -1;
            if (window.RSTH_IH.HobarSlotsIndex >= 0) {
                hindex = window.RSTH_IH.HobarSlotsIndex;
                slot = $gameSystem._customHotbarItems?.[hindex];
                if (slot && slot.type) {
                    if (slot.type === "weapon") {
                        item = window.RSTH_IH.getGameItem(slot);
                        //console.log("item", item);
                        if (item) {
                            atk += item.params[2];
                            matk += item.params[4];
                            //console.log("atk", atk);
                        }
                    }
                    //console.log("slot", slot);
                }
            }

            const hit = window.RSTH_IH.calculateHit(actor.param(5), actor.param(7));
            const eva = window.RSTH_IH.calculateEva(actor.param(6), actor.param(7));
            const cri = window.RSTH_IH.calculateCritical(actor.param(5), actor.param(7));
            const crieva = window.RSTH_IH.calculateCriticalEva(actor.param(6), actor.param(7));
            const aspd = window.RSTH_IH.calculateAtkSpd(actor.param(6));
            const vitplus = actor._paramPlus[3] ?? 0;
            const vit = actor.paramBase(3) + vitplus;

            const stats3 = [
                ["Atk ", atk],
                ["MAtk", matk],
                ["Def ", def],
                ["MDef", mdef]
            ];

            const stats = [
                ["Str", actor.param(2)],
                ["Vit", vit],
                ["Mag", actor.param(4)],
                ["Dex", actor.param(5)],
                ["Agi", actor.param(6)],
                ["Luk", actor.param(7)]
            ];

            const stats2 = [
                ["Hit   ", hit],
                ["Eva   ", eva],
                ["Cri   ", cri],
                ["CriEva", crieva],
                ["AtkSpd", aspd]
            ];

            const fontBackup = this.contents.fontSize;
            this.contents.fontSize = 18;
            const spacing = 20;
            for (let i = 0; i < stats3.length; i++) {
                const [label, value] = stats3[i];
                const y = i * spacing;
                this.drawText(`${label}: ${value}`, 6, y, this.contents.width - 12, 'left');
            }
            for (let i = 0; i < stats.length; i++) {
                const [label, value] = stats[i];
                const y = i * spacing;
                this.drawText(`${label}: ${value}`, 6 + 96, y, this.contents.width - 12, 'left');
            }
            for (let i = 0; i < stats2.length; i++) {
                const [label, value] = stats2[i];
                const y = i * spacing;
                this.drawText(`${label}: ${value}`, 96 * 2, y, this.contents.width - 12, 'left');
            }
            this.contents.fontSize = fontBackup;
        }

        update() {
            super.update();
            this.visible = window.RSTH_IH._showSubHUD;
            this.refresh();
        }
    }

    window.RSTH_IH.ExpBarSprite = class extends Sprite {
        constructor() {
            super(new Bitmap(Graphics.width, 20));
            this.y = Graphics.height - 20;
            this.x = 0;
            this.z = 999; // 前面に表示
            this.opacity = 150;
        }

        update() {
            super.update();

            const actor = $gameParty.leader();
            if (!actor) return;

            const currentLevel = actor.level;
            const expTotal = actor.currentExp();         // 現在の累積EXP
            const expToCurrent = actor.expForLevel(currentLevel);      // 現在レベルの開始EXP
            const expToNext = actor.nextLevelExp();      // 次レベルに必要な累積EXP

            const expInLevel = expTotal - expToCurrent;  // 現在レベル内の進行EXP
            const expNeeded = expToNext - expToCurrent;  // 現在レベルで必要なEXP

            const expRate = expNeeded > 0 ? expInLevel / expNeeded : 1;

            const bw = this.bitmap.width;
            const bh = this.bitmap.height;

            this.bitmap.clear();

            // 背景（黒）
            this.bitmap.fillRect(0, 0, bw, bh, "#000000");

            // 経験値ゲージ（黄系）
            const fillWidth = Math.floor(bw * expRate);
            this.bitmap.gradientFillRect(0, 0, fillWidth, bh, "#ffff66", "#ffaa00");

            // アウトライン付きテキスト描画関数
            const drawTextOutlined = (text, x, y, width, height, align = "left") => {
                const ctx = this.bitmap.context;
                ctx.save();
                ctx.font = this.bitmap._makeFontNameText();
                ctx.textAlign = align;
                ctx.textBaseline = "middle";
                ctx.lineWidth = 6;
                ctx.strokeStyle = "#000000";
                ctx.strokeText(text, x + (align === "center" ? width / 2 : 0), y + height / 2);
                ctx.fillStyle = "#ffffff";
                ctx.fillText(text, x + (align === "center" ? width / 2 : 0), y + height / 2);
                ctx.restore();
            };

            // テキスト（例: Exp. 12 / 100（累計: 250））
            const text = `Exp. ${expInLevel} / ${expNeeded}（累計: ${expTotal}）`;
            this.bitmap.fontSize = 16;
            drawTextOutlined(text, 0, 0, bw, bh, "center");
        }



    };

    window.RSTH_IH.LogWindowManager = {
        _logWindow: null,

        init(scene) {
            const pmargin = 20;
            const gx = Graphics.width;
            const gh = Graphics.height;
            const w = window.RSTH_IH.Hotbarwidth;
            const h = 168;
            const x = gx - window.RSTH_IH.Hotbarwidth - pmargin;
            const y = gh - h - pmargin - 8;
            const rect = new Rectangle(x, y, w, h);
            this._logWindow = new window.RSTH_IH.Window_Log(rect);
            //this._logWindow.opacity = 200; // 透過度を設定
            scene.addWindow(this._logWindow);
        },

        addLog(text, tcolor = 0) {
            if (this._logWindow) {
                this._logWindow.addLog(text, tcolor);
            }
        },

        update() {
            // 必要なら何かあれば
        }
    };

    window.RSTH_IH.Sprite_AnimatedLog = class extends Sprite {
        constructor(text) {
            super(new Bitmap(400, 32));
            this.bitmap.fontSize = 16;
            this.bitmap.outlineWidth = 4;
            this.bitmap.textColor = "rgb(0, 80, 24)";
            this.bitmap.outlineColor = "#ffffff";
            this.bitmap.drawText(text, 0, 0, 400, 32, "right");

            this.x = Graphics.width; // スライドイン開始位置
            this.y = Graphics.height - 32;
            this.targetX = Graphics.width - 420;
            this.targetY = Graphics.height - 170;
            this._age = 0;
            this._isDead = false;
            this.opacity = 0;
        }

        update() {
            this._age++;

            // スライドイン（0〜30）
            if (this._age <= 30) {
                const t = this._age / 30;
                this.x = this.targetX + (1 - t) * 100;
                this.y = this.targetY + (1 - t) * 36;
                this.opacity = Math.min(255, t * 255);
            } else {
                this.x = this.targetX;
                this.y = this.targetY;
            }

            // 600フレーム後フェードアウト
            if (this._age > 600) {
                this.opacity -= 5;
                if (this.opacity <= 0) {
                    this._isDead = true;
                }
            }
        }
    };


    // 青黒い靄のスプライト定義
    window.RSTH_IH.createFogSprite = function () {
        const sprite = new TilingSprite();
        sprite.bitmap = ImageManager.loadBitmap("img/system/", "Fog"); // 画像は img/system/Fog.png と仮定
        sprite.move(0, 0, Graphics.width, Graphics.height);
        sprite.opacity = 96;
        sprite.blendMode = PIXI.BLEND_MODES.ADD;

        // 色調補正（青黒く）
        const colorMatrix = new PIXI.filters.ColorMatrixFilter();
        colorMatrix.brightness(0.5, false); // 少し暗く
        //colorMatrix.tint(0x224488);         // 青黒
        sprite.filters = [colorMatrix];

        // 自動スクロール用
        sprite._fogDx = 0.3 + Math.random() * 0.2;
        sprite._fogDy = 0.1 + Math.random() * 0.1;

        return sprite;
    };


    // Scene_Map に追加と更新を統合
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function () {
        _Scene_Map_createAllWindows.call(this);

        window.RSTH_IH.ACHIEVEMENT_NOTIFIER = new window.RSTH_IH.AchievementNotifier();
        this.addChild(window.RSTH_IH.ACHIEVEMENT_NOTIFIER._container);

        window.RSTH_IH.StatusWindowWidth = 336;
        window.RSTH_IH.StatusWindowHeight = 134;
        const rect1 = new Rectangle(10, 10, window.RSTH_IH.StatusWindowWidth, window.RSTH_IH.StatusWindowHeight);
        this._statusHUD = new Window_StatusHUD(rect1);
        this.addChild(this._statusHUD);

        window.RSTH_IH.StatusSubWindowWidth = 336;
        window.RSTH_IH.StatusSubWindowHeight = 160;
        const rect2 = new Rectangle(10, rect1.y + rect1.height + 10, window.RSTH_IH.StatusSubWindowWidth, window.RSTH_IH.StatusSubWindowHeight);
        this._statusSubHUD = new Window_StatusSubHUD(rect2);
        this.addChild(this._statusSubHUD);

        this._miniMap = new window.RSTH_IH.MiniMapSprite();
        this.addChild(this._miniMap);

        window.RSTH_IH.LogWindowManager.init(this);// ログマネージャ初期化

        this._expBar = new window.RSTH_IH.ExpBarSprite();
        this.addChild(this._expBar);

        this._iconButtons = [];

        const gx = Graphics.width;
        const gh = Graphics.height;
        const buttonSize = 48;
        const margin = 16;
        const baseX = 10;
        const baseY = gh - window.RSTH_IH.Hotbarheight - buttonSize - 48;

        // ステータスサブウィンドウボタン（T）
        const btnStatus = new window.RSTH_IH.ButtonIconSprite("Etc07", () => {
            window.RSTH_IH._showSubHUD = !window.RSTH_IH._showSubHUD;
            if (this._statusSubHUD) this._statusSubHUD.visible = window.RSTH_IH._showSubHUD;

            // 装備ウィンドウ再配置
            if (this._equipmentWindow?.visible) {
                const eqPos = window.RSTH_IH.calculateEquipmentWindowPosition();
                this._equipmentWindow.x = eqPos.x;
                this._equipmentWindow.y = eqPos.y;
            }
        }, "T"); // ← キー表示追加

        btnStatus.x = baseX + 0 * (buttonSize + margin);
        btnStatus.y = baseY;
        this._iconButtons.push(btnStatus);
        this.addChild(btnStatus);

        // 装備ウィンドウボタン（Q）
        const btnEquip = new window.RSTH_IH.ButtonIconSprite("Body03", () => {
            if (this._equipmentWindow) {
                if (this._equipmentWindow.visible) {
                    this._equipmentWindow.hide();
                    this._equipmentWindow.deactivate();
                } else {
                    const eqPos = window.RSTH_IH.calculateEquipmentWindowPosition();
                    this._equipmentWindow.x = eqPos.x;
                    this._equipmentWindow.y = eqPos.y;
                    this._equipmentWindow.show();
                    this._equipmentWindow.activate();
                }
            }
        }, "Q"); // ← キー表示追加

        btnEquip.x = baseX + 1 * (buttonSize + margin);
        btnEquip.y = baseY;
        this._iconButtons.push(btnEquip);
        this.addChild(btnEquip);

        /*
        // インベントリウィンドウボタン（E）
        const btnInventory = new window.RSTH_IH.ButtonIconSprite("Item07", () => {
            if (this._inventoryWindow) {
                if (this._inventoryWindow.visible) {
                    this._inventoryWindow.hide();
                    this._inventoryWindow.deactivate();
                } else {
                    this._inventoryWindow.show();
                    this._inventoryWindow.activate();
                }
            }
        }, "E"); // ← キー表示追加


        btnInventory.x = baseX + 2 * (buttonSize + margin);
        btnInventory.y = baseY;
        this._iconButtons.push(btnInventory);
        this.addChild(btnInventory);
*/

        // SHOPボタン（E）
        const btnShop = new window.RSTH_IH.ButtonIconSpriteNocall("Item07", "E"); // ← キー表示追加
        btnShop.x = baseX + 2 * (buttonSize + margin);
        btnShop.y = baseY;
        this._iconButtons.push(btnShop);
        this.addChild(btnShop);

        // オプションボタン
        const btnOption = new window.RSTH_IH.ButtonIconSpriteNocall("Etc06", "O"); // ← キー表示追加
        btnOption.x = baseX + 3 * (buttonSize + margin);
        btnOption.y = baseY;
        this._iconButtons.push(btnOption);
        this.addChild(btnOption);


        // チュートリアルボタン（L）
        const btnTutorial = new window.RSTH_IH.ButtonIconSpriteNocall("Etc11", "L");
        btnTutorial.x = baseX + 4 * (buttonSize + margin);
        btnTutorial.y = baseY;
        this._iconButtons.push(btnTutorial);
        this.addChild(btnTutorial);


        this._stateIconsHUD = new window.RSTH_IH.Sprite_StateIconsHUD($gameParty.leader());
        this._stateIconsHUD.x = 10;
        this._stateIconsHUD.y = Graphics.boxHeight - 170; // アイコン列の上に配置
        this.addChild(this._stateIconsHUD); // マウスポインタより先に addChild！

        this._stateNamePopup = new window.RSTH_IH.Sprite_StateNamePopup();
        this.addChild(this._stateNamePopup);


        //this.createLevelUpHUD();

        // クエストテキストが存在していたら再表示
        if (window.RSTH_IH._questTextData) {
            this.createQuestTextSprites();
        }

        // ✅ ヴィネットスプライト追加
        this._vignetteSprite = new window.RSTH_IH.VignetteSprite();
        this.addChild(this._vignetteSprite);

        /*
        if (!this._fogSprite) {
            this._fogSprite = window.RSTH_IH.createFogSprite();
            this.addChild(this._fogSprite);
        }
        */
    };

    // Scene_Map 更新内でログも更新
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {

        _Scene_Map_update.call(this);
        window.RSTH_IH.LogWindowManager.update();

        if (Input.isTriggered('t')) {
            window.RSTH_IH._showSubHUD = !window.RSTH_IH._showSubHUD;
            if (this._statusSubHUD) {
                this._statusSubHUD.visible = window.RSTH_IH._showSubHUD;
            }

            // 装備ウィンドウが表示中なら一時的に閉じて、再表示して位置を更新
            if (this._equipmentWindow?.visible) {
                this._equipmentWindow.hide();
                this._equipmentWindow.deactivate();

                // 1フレーム後に再表示して位置再計算
                this._delayReopenEquipWindow = true;
            }
        }

        // 1フレーム後に装備ウィンドウを再表示（位置を動的に再計算）
        if (this._delayReopenEquipWindow) {
            this._delayReopenEquipWindow = false;
            const eqPos = window.RSTH_IH.calculateEquipmentWindowPosition();
            this._equipmentWindow.x = eqPos.x;
            this._equipmentWindow.y = eqPos.y;
            this._equipmentWindow.show();
            this._equipmentWindow.activate();
        }

        if (this._stateIconsHUD) {
            this._stateIconsHUD.refresh();
        }

        if (Input.isTriggered("e")) {
            if (!$gameMessage.isBusy()) {
                window.RSTH_IH.shopCallFlag = 0;
                if (window.RSTH_IH.SetShop === 1) window.RSTH_IH.BuyShop = window.RSTH_IH.ShopGoodsData.basicShop;
                else if (window.RSTH_IH.SetShop === 2) window.RSTH_IH.BuyShop = window.RSTH_IH.ShopGoodsData.uyuyuShop;
                else if (window.RSTH_IH.SetShop === 3) window.RSTH_IH.BuyShop = window.RSTH_IH.ShopGoodsData.pShop;


                //console.log("[Input.isTriggered]window.RSTH_IH.BuyShop", window.RSTH_IH.BuyShop);
                SceneManager.push(window.RSTH_IH.Scene_Shop);
            }
        }

        if (Input.isTriggered("o")) {
            if (!$gameMessage.isBusy()) {
                window.RSTH_IH._calledFromMapScene = true; // マップから来た印
                SceneManager.push(window.RSTH_IH.Scene_OptionsCustom);
            }
        }

        if (Input.isTriggered("l")) {
            if (!$gameMessage.isBusy()) {
                //window.RSTH_IH._calledFromMapScene = true; // マップから来た印
                SceneManager.push(window.RSTH_IH.Scene_Tutorial);
            }
        }

        // 状態異常名ポップアップ処理
        if (this._stateIconsHUD && this._stateNamePopup) {
            const mouseX = TouchInput.x;
            const mouseY = TouchInput.y;
            const hud = this._stateIconsHUD;
            const popup = this._stateNamePopup;

            let found = false;

            for (const sprite of hud._iconSprites) {
                const bx = hud.x + sprite.x;
                const by = hud.y + sprite.y;
                const bw = 32;
                const bh = 32;

                if (
                    mouseX >= bx && mouseX < bx + bw &&
                    mouseY >= by && mouseY < by + bh
                ) {
                    const stateId = sprite._stateId;
                    if (stateId && $dataStates[stateId]) {
                        popup.setStateName($dataStates[stateId].name);
                        found = true;
                    }
                    break;
                }
            }

            if (!found) {
                popup.setStateName(null);
            }
        }


        if (this._vignetteSprite) {
            this._vignetteSprite.update();
        }

    };


    // ログ登録関数（グローバル）
    window.RSTH_IH.logItemGain = function (item, count = 1) {
        if (!item) return;
        const text = `${item.name}を${count}個入手した！`;
        window.RSTH_IH.LogWindowManager.addLog(text);
    };

    window.RSTH_IH.logStatusUP = function (status, amount) {
        const text = `ステータスUPアイテムを入手し、${status}が${amount}UPした！`;
        window.RSTH_IH.LogWindowManager.addLog(text);
    };

    window.RSTH_IH.logGoldGain = function (amount) {
        const text = `${amount}Gを入手した！`;
        window.RSTH_IH.LogWindowManager.addLog(text);
    };

    window.RSTH_IH.logTreasure = function () {
        const text = `トレジャーをGET！！！！`;
        window.RSTH_IH.LogWindowManager.addLog(text, 4);
    };

    window.RSTH_IH.logLvup = function (name, value) {
        const text = `レベルアップで${name}が${value}上昇！！！`;
        window.RSTH_IH.LogWindowManager.addLog(text, 3);
    };

    window.RSTH_IH.logEquipedArmor = function (item) {
        if (!item) return;
        const text = `${item.name}を装備した！`;
        window.RSTH_IH.LogWindowManager.addLog(text);
    };

    window.RSTH_IH.logSelledArmor = function (item, remaining = 1) {
        if (!item) return;
        const text = `${item.name}x${remaining}を売却した！`;
        window.RSTH_IH.LogWindowManager.addLog(text);
    };

    // ✅ ログ用ウィンドウ
    window.RSTH_IH.Window_Log = class extends Window_Base {
        constructor(rect) {
            super(rect);
            this._lines = [];
            this._flags = [];
            this.maxLines = 20; // 表示上限行数
            this.lineHeightEx = this.lineHeight();

            // ✅ 枠を完全に無効化
            this.opacity = 0;          // 外枠とパディングの透明化
            this.backOpacity = 0;      // デフォルト背景も無効化
            this.setBackgroundType(2); // RPGツクールの背景種別は透明（0:通常, 1:透明, 2:暗色）

            // ✅ 背景スプライトで黒背景を自前で生成
            this.createBackgroundSprite();

            this.refresh();
        }

        createBackgroundSprite() {
            const w = this.width;
            const h = this.height;
            const bitmap = new Bitmap(w, h);
            bitmap.fillAll("#000000");       // 背景を黒で塗りつぶす
            this._backgroundSprite = new Sprite(bitmap);
            this._backgroundSprite.alpha = 0.4; // 透過度(0〜1)
            this.addChildToBack(this._backgroundSprite);
        }

        lineHeight() {
            return this.contents.fontSize + 2;
        }

        resetFontSettings() {
            this.contents.fontSize = 20;
            this.resetTextColor();
        }

        addLog(text, tcolor = 0) {
            this._lines.push(text);
            this._flags.push(tcolor);
            if (this._lines.length > this.maxLines) {
                this._lines.shift();
                this._flags.shift();
            }
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            const lineH = this.lineHeightEx;
            const margin = 0;

            const maxVisibleLines = Math.floor((this.contentsHeight() - margin * 2) / lineH);
            const linesToDraw = this._lines.slice(-maxVisibleLines);
            const flagsToDraw = this._flags.slice(-maxVisibleLines);

            const totalLinesHeight = linesToDraw.length * lineH;
            let y = this.contentsHeight() - totalLinesHeight - margin;

            for (let i = 0; i < linesToDraw.length; i++) {
                const text = linesToDraw[i];
                const flag = flagsToDraw[i] ?? 0;

                //console.log("flag", flag);
                if (flag === 0) {
                    this.resetTextColor();
                }
                else if (flag === 1) {//boss倒した
                    this.changeTextColor(ColorManager.crisisColor());
                }
                else if (flag === 2) {//うんしょp獲得
                    this.changeTextColor("rgba(147, 255, 162, 1)");
                }
                else if (flag === 3) {//LvUP
                    this.changeTextColor("rgba(230, 255, 3, 1)");
                }
                else if (flag === 4) {//トレジャー獲得
                    this.changeTextColor("rgba(174, 180, 255, 1)");
                }
                else {
                    this.changeTextColor("rgba(255, 0, 0, 1)");
                }

                this.drawText(text, margin, y + margin, this.contentsWidth() - margin * 2, "left");
                //this.drawTextEx(text, margin, y + margin);
                y += lineH;
            }
        }
    };


    // ✅ ヴィネットスプライトクラス
    window.RSTH_IH.VignetteSprite = class extends Sprite {
        constructor() {
            super(new Bitmap(Graphics.width, Graphics.height));
            this._isVisible = false;
            this.opacity = 0;
            this.z = 1000;

            this._drawVignette();
        }

        _drawVignette() {
            const ctx = this.bitmap.context;
            const w = this.bitmap.width;
            const h = this.bitmap.height;

            const gradient = ctx.createRadialGradient(w / 2, h / 2, Math.min(w, h) / 4, w / 2, h / 2, Math.max(w, h) / 1.2);
            gradient.addColorStop(0, "rgba(0,0,0,0)");
            gradient.addColorStop(1, "rgba(255, 0, 0, 0.6)");

            ctx.fillStyle = gradient;
            ctx.fillRect(0, 0, w, h);
            this.bitmap._baseTexture.update(); // テクスチャ更新
        }

        update() {
            super.update();
            const actor = $gameParty.leader();
            if (!actor) return;

            const ratio = actor.hp / actor.mhp;

            if (ratio <= 0.25) {
                this._isVisible = true;
                this.opacity = Math.min(255, this.opacity + 10);
            } else {
                this._isVisible = false;
                this.opacity = Math.max(0, this.opacity - 10);
            }
        }
    };

    window.RSTH_IH.showScreenBorder = function (flag = 0) {
        const scene = SceneManager._scene;
        if (!scene || !scene._spriteset) return;

        const borderSprites = [];

        let color = "#fffb0096";
        if (flag === 1) {
            color = "#002fff96";
        }
        const thickness = 24;     // 枠の太さ

        const width = Graphics.width;
        const height = Graphics.height;

        // 上辺
        const top = new Sprite();
        top.bitmap = new Bitmap(width, thickness);
        top.bitmap.fillAll(color);
        top.x = 0;
        top.y = 0;
        borderSprites.push(top);

        // 下辺
        const bottom = new Sprite();
        bottom.bitmap = new Bitmap(width, thickness);
        bottom.bitmap.fillAll(color);
        bottom.x = 0;
        bottom.y = height - thickness;
        borderSprites.push(bottom);

        // 左辺
        const left = new Sprite();
        left.bitmap = new Bitmap(thickness, height);
        left.bitmap.fillAll(color);
        left.x = 0;
        left.y = 0;
        borderSprites.push(left);

        // 右辺
        const right = new Sprite();
        right.bitmap = new Bitmap(thickness, height);
        right.bitmap.fillAll(color);
        right.x = width - thickness;
        right.y = 0;
        borderSprites.push(right);

        // 追加＆1秒後に削除
        for (const sprite of borderSprites) {
            sprite.z = 999;
            scene._spriteset.addChild(sprite);
        }

        setTimeout(() => {
            for (const sprite of borderSprites) {
                scene._spriteset.removeChild(sprite);
            }
        }, 1000);
    };

    // アイコン
    window.RSTH_IH.ButtonIconSprite = class extends Sprite {
        constructor(filename, callback, keyLabel = "") {
            const bitmap = ImageManager.loadSystem(filename);
            super(bitmap);

            this._callback = callback;
            this._clicking = false;

            // 🔽 テキスト表示用スプライト（画像とは別）
            if (keyLabel) {
                this._labelSprite = new window.RSTH_IH.LabelTextSprite(keyLabel);
                this._labelSprite.x = 16; // ※相対位置（右下に調整）
                this._labelSprite.y = 32;
                this.addChild(this._labelSprite);
            }

            this.scale.x = 0.8;
            this.scale.y = 0.8;

            this.interactive = true;
            this.buttonMode = true;
            this.anchor.x = 0;
            this.anchor.y = 0;
        }

        update() {
            super.update();
            if (TouchInput.isPressed() && this.hitTest(TouchInput.x, TouchInput.y)) {
                if (!this._clicking) {
                    this._clicking = true;
                    if (this._callback) this._callback();
                }
            } else {
                this._clicking = false;
            }
        }

        hitTest(x, y) {
            const localX = x - this.x;
            const localY = y - this.y;
            return localX >= 0 && localX < this.width && localY >= 0 && localY < this.height;
        }
    };

    // アイコン　コールバックなし
    window.RSTH_IH.ButtonIconSpriteNocall = class extends Sprite {
        constructor(filename, keyLabel = "") {
            const bitmap = ImageManager.loadSystem(filename);
            super(bitmap);

            this._clicking = false;

            // 🔽 テキスト表示用スプライト（画像とは別）
            if (keyLabel) {
                this._labelSprite = new window.RSTH_IH.LabelTextSprite(keyLabel);
                this._labelSprite.x = 16; // ※相対位置（右下に調整）
                this._labelSprite.y = 32;
                this.addChild(this._labelSprite);
            }

            this.scale.x = 0.8;
            this.scale.y = 0.8;

            this.interactive = true;
            this.buttonMode = true;
            this.anchor.x = 0;
            this.anchor.y = 0;
        }

        update() {
            super.update();
            if (TouchInput.isPressed() && this.hitTest(TouchInput.x, TouchInput.y)) {
                if (!this._clicking) {
                    this._clicking = true;
                }
            } else {
                this._clicking = false;
            }
        }

        hitTest(x, y) {
            const localX = x - this.x;
            const localY = y - this.y;
            return localX >= 0 && localX < this.width && localY >= 0 && localY < this.height;
        }
    };

    // 🔧 テキスト表示用スプライトクラス
    window.RSTH_IH.LabelTextSprite = class extends Sprite {
        constructor(text) {
            const bitmap = new Bitmap(64, 64); // 必要に応じてサイズ調整
            super(bitmap);
            this.draw(text);
        }

        draw(text) {
            const size = 24;
            this.bitmap.fontSize = size;
            this.bitmap.textColor = "#ffffff";
            this.bitmap.outlineColor = "#000000";
            this.bitmap.outlineWidth = 6;

            // 右下に描画
            this.bitmap.drawText(text, 0, 0, this.bitmap.width, this.bitmap.height, "center");
        }
    };

    // 状態異常アイコン表示
    window.RSTH_IH.Sprite_StateIconsHUD = class extends Sprite {
        constructor(actor) {
            super();
            this._actor = actor;
            this.z = 10;
            this.opacity = 230;
            this.anchor.x = 0;
            this.anchor.y = 0;
            this._iconSprites = [];
            this.refresh();
        }

        refresh() {
            // 既存スプライト削除
            for (const sprite of this._iconSprites) {
                this.removeChild(sprite);
            }
            this._iconSprites = [];

            const icons = this._actor ? this._actor.allIcons() : [];
            const states = this._actor ? this._actor.states() : [];
            const iconWidth = 32;
            const iconHeight = 32;
            const displaySize = 36;
            const scale = displaySize / iconWidth;

            const iconsPerRow = 8;
            const totalRows = Math.ceil(icons.length / iconsPerRow);

            for (let i = 0; i < icons.length; i++) {
                const iconId = icons[i];
                const sprite = new Sprite();
                sprite.bitmap = ImageManager.loadSystem("IconSet");
                sprite.setFrame(
                    (iconId % 16) * iconWidth,
                    Math.floor(iconId / 16) * iconHeight,
                    iconWidth,
                    iconHeight
                );

                const spacing = 5;
                const col = i % iconsPerRow;
                const row = Math.floor(i / iconsPerRow);
                sprite.x = col * (displaySize + spacing);
                sprite.y = this.height - (displaySize + spacing) * (totalRows - row);

                sprite.scale.set(scale, scale);

                // ステートIDをスプライトに保持
                const state = states.find(s => s.iconIndex === iconId);
                sprite._stateId = state?.id ?? null;

                sprite.opacity = this.opacity;
                this._iconSprites.push(sprite);
                this.addChild(sprite);
            }
        }

        update() {
            super.update();
            // 自動更新が必要ならここに状態チェック処理を追加
        }
    };


    const _Game_Battler_addNewState = Game_Battler.prototype.addNewState;
    Game_Battler.prototype.addNewState = function (stateId) {
        const actor = this;
        const state = $dataStates[stateId];
        const message = state.message1; // ステート付加時のメッセージ

        _Game_Battler_addNewState.call(this, stateId);

        // 条件：アクターで、マップ上で、メッセージが設定されている場合
        if (actor.isActor() && SceneManager._scene instanceof Scene_Map && message) {
            const name = actor.name();
            const text = message.replace("%1", name);
            window.RSTH_IH.LogWindowManager.addLog(text);
        }
    };

    window.RSTH_IH.Sprite_StateNamePopup = class extends Sprite {
        constructor() {
            super(new Bitmap(1, 1));
            this.visible = false;
            this.anchor.set(0, 1);
            this.padding = 6;
            this.lineHeight = 24;
        }

        setStateName(name) {
            const bmp = this.bitmap;
            bmp.clear();

            if (!name) {
                this.visible = false;
                return;
            }

            const textWidth = bmp.measureTextWidth(name); // ← 修正済み
            const w = textWidth + this.padding * 2;
            const h = this.lineHeight + this.padding * 2;

            this.bitmap = new Bitmap(w, h);
            const b = this.bitmap;

            b.fontSize = 18;
            b.textColor = "#ffffff";
            b.outlineColor = "#000000";
            b.outlineWidth = 3;

            b.fillRect(0, 0, w, h, "rgba(0,0,0,0.7)");
            b.drawText(name, this.padding, this.padding, w - this.padding * 2, this.lineHeight, "left");

            this.visible = true;
        }


        update() {
            super.update();
            if (this.visible) {
                const offsetX = 16;
                const offsetY = 16;
                const bw = this.bitmap.width;
                const bh = this.bitmap.height;

                let newX = TouchInput.x + offsetX;
                let newY = TouchInput.y + offsetY;

                // 吸着処理
                newX = Math.max(0, Math.min(Graphics.width - bw, newX));
                newY = Math.max(bh, Math.min(Graphics.height, newY));

                this.x = newX;
                this.y = newY;
            }
        }
    };

    Game_Player.prototype.startHitStop = function (frameCount = 5) {
        this._hitStopCount = frameCount;
    };

    const _Game_Player_update = Game_Player.prototype.update;
    Game_Player.prototype.update = function (sceneActive) {
        if (this._hitStopCount > 0) {
            this._hitStopCount--;
            return; // ヒットストップ中は何もしない（移動・入力含む完全停止）
        }
        _Game_Player_update.call(this, sceneActive);
    };

    // 邪魔なメッセージを非表示
    Game_Actor.prototype.showAddedStates = function () {
        /*
        for (const state of this.result().addedStateObjects()) {
            if (state.message1) {
                $gameMessage.add(state.message1.format(this._name));
            }
        }
            */
    };


})();