/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {

    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;

    window.RSTH_IH.Scene_Shop = class extends Scene_MenuBase {
        constructor() {
            super();
            this.margin = 480;
        }

        initialize() {
            super.initialize();
        }

        create() {
            super.create();
            //this.createTitle();
            this.createHelpWindow();
            this.createFaceMessageWindow();
            this.createGoldWindow();
            this.createCommandWindow();
            this.createDummyWindow();
            this.createNumberWindow();
            this.createStatusWindow();
            this.createBuyWindow();
            this.createShopCategoryWindow();
            this.createCategoryWindow();
            this.createSellWindow();
            this.createCannotCarryWindow();
        }

        createTitle() {
            this._title = new window.RSTH_IH.Sprite_VerticalTitleText("Shop");
            this.addChild(this._title);
        }

        mainAreaTop() {
            return 196;
        }

        mainAreaHeight() {
            const gh = Graphics.height;
            const ah = gh - this.mainAreaTop() - this.helpAreaHeight() - 6;
            return ah
        }

        createFaceMessageWindow() {
            const wx = this.margin;  // ← メッセージウィンドウは余白分右へ
            const wy = 0;
            const ww = Graphics.boxWidth - wx;
            const wh = this.mainAreaTop();
            const rect = new Rectangle(wx, wy, ww, wh);

            let facetext = "";
            let pic = "";
            if (window.RSTH_IH.shopCallFlag === 1) {
                pic = "Package2_7";
                facetext = "次元間商人チムグ\n「次元間ショップです。\n　今回のみ半額セールを実施中です！」";
            } else {
                pic = "Package1_6";
                facetext = "次元間商人ディララ\n「この度は当次元間ショップをお呼びしていただき\n　まことにありがとうございます。\n　さて、今回はどの商品を購入されますか？」";
            }

            this._faceMessageWindow = new window.RSTH_IH.Window_FaceMessage(rect);

            this._faceMessageWindow.setMessageText(facetext);
            this.addWindow(this._faceMessageWindow);

            // 立ち絵スプライトを余白部分に表示

            const standSprite = new Sprite(ImageManager.loadPicture(pic));
            standSprite.x = 0; // 余白の左端に表示
            standSprite.y = 0; // 必要なら位置調整

            standSprite.scale.set(2, 2);

            if (window.RSTH_IH.shopCallFlag === 1) standSprite.anchor.x = 0.73;
            else standSprite.anchor.x = 0.83;

            standSprite.anchor.y = 0;

            this._faceMessageWindow.addChild(standSprite);
            //this.addChild(standSprite);
        }

        createGoldWindow() {
            const rect = this.goldWindowRect();
            this._goldWindow = new Window_Gold(rect);
            this.addWindow(this._goldWindow);
        }

        goldWindowRect() {
            const ww = this.mainCommandWidth();
            const wh = this.calcWindowHeight(1, true);
            const wx = Graphics.boxWidth - ww;
            const wy = this.mainAreaTop();
            return new Rectangle(wx, wy, ww, wh);
        }

        createCommandWindow() {
            const rect = this.commandWindowRect();
            this._commandWindow = new window.RSTH_IH.Window_ShopCommand(rect);
            this._purchaseOnly = true;
            this._commandWindow.setPurchaseOnly(this._purchaseOnly);
            this._commandWindow.y = this.mainAreaTop();
            this._commandWindow.setHandler("buy", this.commandBuy.bind(this));
            this._commandWindow.setHandler("cancel", () => {
                SceneManager.goto(Scene_Map);
            });
            this.addWindow(this._commandWindow);
        }


        commandWindowRect() {
            const wx = this.margin;
            const wy = this.mainAreaTop();
            const ww = this._goldWindow.x - wx;
            const wh = this.calcWindowHeight(1, true);
            return new Rectangle(wx, wy, ww, wh);
        }

        createDummyWindow() {
            const rect = this.dummyWindowRect();
            this._dummyWindow = new Window_Base(rect);
            this.addWindow(this._dummyWindow);
        }

        dummyWindowRect() {
            const wx = this.margin;
            const wy = this._commandWindow.y + this._commandWindow.height + this.calcWindowHeight(1, true);
            const ww = Graphics.boxWidth - wx;
            const wh = this.mainAreaHeight() - this._commandWindow.height - this.calcWindowHeight(1, true);
            return new Rectangle(wx, wy, ww, wh);
        }

        createNumberWindow() {
            const rect = this.numberWindowRect();
            this._numberWindow = new window.RSTH_IH.Window_ShopNumber(rect);
            this._numberWindow.hide();
            this._numberWindow.setHandler("ok", this.onNumberOk.bind(this));
            this._numberWindow.setHandler("cancel", this.onNumberCancel.bind(this));
            this.addWindow(this._numberWindow);
        }

        numberWindowRect() {
            const wx = this.margin;
            const wy = this._dummyWindow.y;
            const ww = Graphics.boxWidth - this.statusWidth() - wx;
            const wh = this._dummyWindow.height;
            return new Rectangle(wx, wy, ww, wh);
        }

        createStatusWindow() {
            const rect = this.statusWindowRect();
            this._statusWindow = new window.RSTH_IH.Window_ShopStatus(rect);
            this._statusWindow.hide();
            this.addWindow(this._statusWindow);
        }

        statusWindowRect() {
            const ww = this.statusWidth();
            const wh = this._dummyWindow.height;
            const wx = Graphics.boxWidth - ww;
            const wy = this._dummyWindow.y;
            return new Rectangle(wx, wy, ww, wh);
        }

        createBuyWindow() {
            const rect = this.buyWindowRect();
            this._buyWindow = new window.RSTH_IH.Window_ShopBuy(rect);
            this._buyWindow.setupGoods(window.RSTH_IH.BuyShop);

            // console.log("window.RSTH_IH.BuyShop", window.RSTH_IH.BuyShop);
            this._buyWindow.setHelpWindow(this._helpWindow);
            this._buyWindow.setStatusWindow(this._statusWindow);
            this._buyWindow.hide();
            this._buyWindow.setHandler("ok", this.onBuyOk.bind(this));
            this._buyWindow.setHandler("cancel", this.onBuyCancel.bind(this));
            this.addWindow(this._buyWindow);
        }

        buyWindowRect() {
            const wx = this.margin;
            const wy = this._dummyWindow.y;
            const ww = Graphics.boxWidth - this.statusWidth() - wx;
            const wh = this._dummyWindow.height;
            return new Rectangle(wx, wy, ww, wh);
        }

        helpWindowRect() {
            const wx = this.margin;
            const wy = this.helpAreaTop();
            const ww = Graphics.boxWidth - wx;
            const wh = this.helpAreaHeight();
            return new Rectangle(wx, wy, ww, wh);
        }

        update() {
            super.update();
            if (this._cannotCarryWindow && this._cannotCarryWindow.visible) {
                if (Input.isTriggered("ok") || Input.isTriggered("cancel")) {
                    this._cannotCarryWindow.hide();
                    this._numberWindow.activate();
                }
            }
        }

        createShopCategoryWindow() {
            const rect = this.categoryWindowRect();
            this._shopCategoryWindow = new window.RSTH_IH.Window_ShopCategory(rect);
            this._shopCategoryWindow.setHandler("ok", this.onShopCategoryOk.bind(this));
            this._shopCategoryWindow.setHandler("cancel", this.onShopCategoryCancel.bind(this));
            this._shopCategoryWindow.show();
            this._shopCategoryWindow.deactivate();
            this.addWindow(this._shopCategoryWindow);
        }


        createCategoryWindow() {
            const rect = this.categoryWindowRect();
            this._categoryWindow = new window.RSTH_IH.Window_ItemCategory(rect);
            this._categoryWindow.setHelpWindow(this._helpWindow);
            this._categoryWindow.hide();
            this._categoryWindow.deactivate();
            this._categoryWindow.setHandler("ok", this.onCategoryOk.bind(this));
            this._categoryWindow.setHandler("cancel", this.onCategoryCancel.bind(this));
            this.addWindow(this._categoryWindow);
        }

        categoryWindowRect() {
            const wx = this.margin;
            const wy = this._dummyWindow.y - this.calcWindowHeight(1, true);
            const ww = Graphics.boxWidth - wx;
            const wh = this.calcWindowHeight(1, true);
            return new Rectangle(wx, wy, ww, wh);
        }

        createSellWindow() {
            const rect = this.sellWindowRect();
            this._sellWindow = new window.RSTH_IH.Window_ShopSell(rect);
            this._sellWindow.setHelpWindow(this._helpWindow);
            this._sellWindow.hide();
            this._sellWindow.setHandler("ok", this.onSellOk.bind(this));
            this._sellWindow.setHandler("cancel", this.onSellCancel.bind(this));
            this._categoryWindow.setItemWindow(this._sellWindow);
            this.addWindow(this._sellWindow);
            if (!this._categoryWindow.needsSelection()) {
                this._sellWindow.y -= this._categoryWindow.height;
                this._sellWindow.height += this._categoryWindow.height;
            }
        }

        sellWindowRect() {
            const wx = this.margin;
            const wy = this._categoryWindow.y + this._categoryWindow.height;
            const ww = Graphics.boxWidth;
            const wh = this.mainAreaHeight() - this._commandWindow.height - this._categoryWindow.height;
            return new Rectangle(wx, wy, ww, wh);
        }

        statusWidth() {
            return 352;
        }

        activateBuyWindow() {
            this._buyWindow.setMoney(this.money());
            this._buyWindow.show();
            this._buyWindow.activate();
            this._statusWindow.show();
        }

        activateSellWindow() {
            if (this._categoryWindow.needsSelection()) {
                this._categoryWindow.show();
            }
            this._sellWindow.refresh();
            this._sellWindow.show();
            this._sellWindow.activate();
            this._statusWindow.hide();
        }

        commandBuy() {
            //this._dummyWindow.hide();
            this._shopCategoryWindow.activate();
            // this.activateBuyWindow();
        }

        commandSell() {
            this._dummyWindow.hide();
            this._sellWindow.show();
            this._sellWindow.deselect();
            this._sellWindow.refresh();
            if (this._categoryWindow.needsSelection()) {
                this._categoryWindow.show();
                this._categoryWindow.activate();
            } else {
                this.onCategoryOk();
            }
        }

        onShopCategoryOk() {
            this._dummyWindow.hide();
            const category = this._shopCategoryWindow.currentSymbol();

            const filteredGoods = window.RSTH_IH.BuyShop.filter(goods => {
                if (category === "item") return goods[0] === 0;
                if (category === "weapon") return goods[0] === 1;
                if (category === "armor") return goods[0] === 2;
                return true;
            });
            this._buyWindow.setupGoods(filteredGoods);
            //this._shopCategoryWindow.hide();
            this.activateBuyWindow();
        }


        onShopCategoryCancel() {
            //this._shopCategoryWindow.hide();
            this._dummyWindow.show();
            this._commandWindow.activate();
        }

        onBuyOk() {
            this._item = this._buyWindow.item();
            this._buyWindow.hide();
            this._numberWindow.setup(this._item, this.maxBuy(), this.buyingPrice());
            this._numberWindow.setCurrencyUnit(this.currencyUnit());
            this._numberWindow.show();
            this._numberWindow.activate();
        }

        onBuyCancel() {
            this._shopCategoryWindow.activate();
            //this._commandWindow.activate();
            this._dummyWindow.show();
            this._buyWindow.hide();
            this._statusWindow.hide();
            this._statusWindow.setItem(null);
            this._helpWindow.clear();
        }

        onCategoryOk() {
            this.activateSellWindow();
            this._sellWindow.select(0);
        }

        onCategoryCancel() {
            this._commandWindow.activate();
            this._dummyWindow.show();
            this._categoryWindow.hide();
            this._sellWindow.hide();
        }

        onSellOk() {
            this._item = this._sellWindow.item(); // ここで slot を取得
            this._categoryWindow.hide();
            this._sellWindow.hide();

            this._numberWindow.setup(
                this._item,
                this.maxSell(),
                this.sellingPrice() // 修正後の sellingPrice() を呼ぶ
            );
            this._numberWindow.setCurrencyUnit(this.currencyUnit());
            this._numberWindow.show();
            this._numberWindow.activate();

            const resolvedItem = this.resolveItem(this._item);
            this._statusWindow.setItem(resolvedItem);
            this._statusWindow.show();
        }

        onSellCancel() {
            this._sellWindow.deselect();
            this._statusWindow.setItem(null);
            this._helpWindow.clear();
            if (this._categoryWindow.needsSelection()) {
                this._categoryWindow.activate();
            } else {
                this.onCategoryCancel();
            }
        }

        onNumberOk() {
            switch (this._commandWindow.currentSymbol()) {
                case "buy":
                    if (window.RSTH_IH.isInventoryOrHotbarFull()) {
                        SoundManager.playBuzzer();
                        this.showCannotCarryMessage();
                        return; // 入力終了しない
                    } else {
                        SoundManager.playShop();
                        this.doBuy(this._numberWindow.number());

                    }
                    break;
                case "sell":
                    SoundManager.playShop();
                    this.doSell(this._numberWindow.number());
                    break;
            }
            this.endNumberInput();
            this._goldWindow.refresh();
            this._statusWindow.refresh();
        }

        onNumberCancel() {
            SoundManager.playCancel();
            this.endNumberInput();
        }

        createCannotCarryWindow() {
            const rect = new Rectangle(
                Graphics.boxWidth / 2 - 150,
                Graphics.boxHeight / 2 - 40,
                300,
                this.calcWindowHeight(2, true)
            );
            this._cannotCarryWindow = new Window_Base(rect);
            this._cannotCarryWindow.hide();
            this._cannotCarryWindow.activate = function () { };
            this.addWindow(this._cannotCarryWindow);
        }

        showCannotCarryMessage() {
            this._cannotCarryWindow.contents.clear();
            this._cannotCarryWindow.drawText(
                "これ以上持てません",
                0,
                0,
                this._cannotCarryWindow.contentsWidth(),
                "center"
            );
            this._cannotCarryWindow.show();
            this._cannotCarryWindow.activate();
        }


        doBuy(number) {
            const item = this._item;
            const goods = window.RSTH_IH.BuyShop.find(g => {
                const type = this.getItemType(item);
                return g[0] === (type === "item" ? 0 : type === "weapon" ? 1 : 2) && g[1] === item.id;
            });

            if (goods && goods.length >= 5) {
                const stockKey = `${this.getItemType(item)}_${item.id}`;
                if (!$gameSystem._shopStocks) $gameSystem._shopStocks = {};
                if ($gameSystem._shopStocks[stockKey] == null) {
                    $gameSystem._shopStocks[stockKey] = goods[4]; // 初期化
                }
                $gameSystem._shopStocks[stockKey] -= number;
                if ($gameSystem._shopStocks[stockKey] < 0) {
                    $gameSystem._shopStocks[stockKey] = 0;
                }
            }

            // 在庫を反映するためにデータを再構築してリフレッシュ
            this._buyWindow._data = null;
            this._buyWindow.makeItemList();
            this._buyWindow.refresh();

            $gameParty.loseGold(number * this.buyingPrice());
            $gameParty.gainItem(item, number);

            if (this._faceMessageWindow) {
                const displayName = item.name || "???";

                let facetext = "";
                if (window.RSTH_IH.shopCallFlag === 1) {
                    facetext = `次元間商人チムグ\n「${displayName} を\n　${number}個購入していただき、\n　ありがとうございました！」`;;
                } else {
                    facetext = `次元間商人ディララ\n「${displayName} を\n　${number}個購入していただき、\n　ありがとうございました！」`;
                }

                this._faceMessageWindow.setMessageText(facetext);
            }
        }



        endNumberInput() {
            this._numberWindow.hide();
            switch (this._commandWindow.currentSymbol()) {
                case "buy":
                    this.activateBuyWindow();
                    break;
                case "sell":
                    this.activateSellWindow();
                    break;
            }
        }

        maxBuy() {
            const item = this._item;
            if (!item) return 0;

            const price = this.buyingPrice();
            const moneyMax = price > 0 ? Math.floor(this.money() / price) : Infinity;

            const inv = $gameSystem._customInventoryItems || [];
            const hot = $gameSystem._customHotbarItems || [];

            const type = this.getItemType(item);
            let maxStack = DataManager.isItem(item) ? window.RSTH_IH.StackSize || 1 : 1;
            let totalSpace = 0;

            const countSlots = (slots) => {
                for (const slot of slots) {
                    if (!slot) {
                        totalSpace += maxStack;
                    } else if (slot.id === item.id && slot.type === type) {
                        const remain = maxStack - (slot.count || 0);
                        if (remain > 0) totalSpace += remain;
                    }
                }
            };

            countSlots(inv);
            countSlots(hot);

            // 在庫制限を取得
            const goods = window.RSTH_IH.BuyShop.find(g => {
                return g[0] === (type === "item" ? 0 : type === "weapon" ? 1 : 2) && g[1] === item.id;
            });

            let stock = Infinity;
            if (goods && goods.length >= 5) {
                const stockKey = `${type}_${item.id}`;
                if (!$gameSystem._shopStocks) $gameSystem._shopStocks = {};
                if ($gameSystem._shopStocks[stockKey] == null) {
                    $gameSystem._shopStocks[stockKey] = goods[4];
                }
                stock = $gameSystem._shopStocks[stockKey];
            }

            return Math.min(moneyMax, totalSpace, stock);
        }



        getItemType(item) {
            if (DataManager.isItem(item)) return "item";
            if (DataManager.isWeapon(item)) return "weapon";
            if (DataManager.isArmor(item)) return "armor";
            return null;
        }


        money() {
            return this._goldWindow.value();
        }

        currencyUnit() {
            return this._goldWindow.currencyUnit();
        }

        buyingPrice() {
            return this._buyWindow.price(this._item);
        }

        sellingPrice() {
            const slot = this._item;
            const item = this.resolveItem(slot);
            //console.log("item.price", item.price);
            return item ? Math.floor((item.price || 0) / 5) : 0;
        }

        resolveItem(slot) {
            if (!slot || slot.empty) return null;
            switch (slot.type) {
                case "item": return $dataItems[slot.id];
                case "weapon": return $dataWeapons[slot.id];
                case "armor": return $dataArmors[slot.id];
                default: return null;
            }
        }

        maxSell() {
            const slot = this._item;
            return slot && !slot.empty ? slot.count : 0;
        }

        doSell(number) {
            const slot = this._item;
            const item = this.resolveItem(slot);
            if (!item) return;

            const goldGain = number * this.sellingPrice();
            $gameParty.gainGold(goldGain);

            // 実際にスロットから個数を減らす（例）
            slot.count -= number;
            if (slot.count <= 0) {
                const inv = $gameSystem._customInventoryItems;
                const hot = $gameSystem._customHotbarItems;
                if (this._categoryWindow.currentSymbol() === "inventory") {
                    inv[slot.__slotIndex] = null;
                } else if (this._categoryWindow.currentSymbol() === "hotbar") {
                    hot[slot.__slotIndex] = null;
                }
            }

            //console.log(`Sold: ${item.name} x${number} for ${goldGain}G`);
        }

    };

    window.RSTH_IH.Window_ShopCommand = class extends Window_HorzCommand {
        constructor(rect) {
            super(rect);
            this._purchaseOnly = true;
        }

        setPurchaseOnly(purchaseOnly) {
            this._purchaseOnly = purchaseOnly;
            this.refresh();
        }

        maxCols() {
            return 2; // ← 2等分！
        }

        makeCommandList() {
            this.addCommand(TextManager.buy, "buy");
            // 売却を追加しない
            // this.addCommand(TextManager.sell, "sell", !this._purchaseOnly);
            this.addCommand(TextManager.cancel, "cancel");
        }
    };


    window.RSTH_IH.Window_FaceMessage = class extends Window_Base {
        constructor(rect) {
            super(rect);
            this._faceName = "";
            this._faceIndex = 0;
            this._messageText = "";
            this._faceLoaded = false;
            this._margin = 12;

            // 顔グラはロード完了を監視
            this._faceBitmap = null;
        }

        setFaceName(name, index) {
            this._faceName = name;
            this._faceIndex = index;

            if (name) {
                this._faceBitmap = ImageManager.loadFace(name);
                this._faceBitmap.addLoadListener(() => {
                    this._faceLoaded = true;
                    this.refresh();
                });
            }
        }

        setMessageText(text) {
            this._messageText = text;
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            const indent = 36; // 左から20px空けたい
            const textX = this._margin + indent;
            const textY = this._margin;

            this.drawTextEx(this._messageText, textX, textY);
        }


    };

    window.RSTH_IH.Window_ShopBuy = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this._money = 0;
            this._shopGoods = [];
        }

        setupGoods(shopGoods) {
            this._shopGoods = [];
            this._shopGoods = shopGoods;
            //console.log("this._shopGoods", this._shopGoods);
            this.refresh();
            this.select(0);
        }

        maxItems() {
            return this._data ? this._data.length : 1;
        }

        item() {
            return this.itemAt(this.index());
        }

        itemAt(index) {
            return this._data && index >= 0 ? this._data[index] : null;
        }

        setMoney(money) {
            this._money = money;
            this.refresh();
        }

        isCurrentItemEnabled() {
            return this.isEnabled(this._data[this.index()]);
        }

        price(item) {
            return this._price[this._data.indexOf(item)] || 0;
        }

        getStockKey(item) {
            if (DataManager.isItem(item)) return `item_${item.id}`;
            if (DataManager.isWeapon(item)) return `weapon_${item.id}`;
            if (DataManager.isArmor(item)) return `armor_${item.id}`;
            return `unknown_${item.id}`;
        }

        isEnabled(item) {
            if (!item) return false;

            const price = this.price(item);
            const moneyOk = price <= this._money;
            const currentCount = window.RSTH_IH.getCustomNumItems(item);

            let isMaxed = false;
            if (DataManager.isItem(item)) {
                const maxAllowed = (window.RSTH_IH.StackSize || 1) *
                    (window.RSTH_IH.InventoryCols * window.RSTH_IH.InventoryRows + window.RSTH_IH.HotbarSlotCount);
                isMaxed = currentCount >= maxAllowed;
            } else {
                const totalSlots = window.RSTH_IH.InventoryCols * window.RSTH_IH.InventoryRows + window.RSTH_IH.HotbarSlotCount;
                isMaxed = currentCount >= totalSlots;
            }

            const stockKey = this.getStockKey(item);
            const stock = $gameSystem._shopStocks?.[stockKey] ?? -1;
            if (stock === 0) return false;

            return moneyOk && !isMaxed;
        }

        refresh() {
            this.makeItemList();
            super.refresh();
            this.contents.clear();
            this.drawAllItems();
        }

        makeItemList() {
            this._data = [];
            this._price = [];
            for (const goods of this._shopGoods) {
                const item = this.goodsToItem(goods);
                if (item) {
                    const stockKey = this.getStockKey(item);
                    if (!$gameSystem._shopStocks) $gameSystem._shopStocks = {};
                    if (!(stockKey in $gameSystem._shopStocks)) {
                        if (goods.length >= 5) {
                            $gameSystem._shopStocks[stockKey] = goods[4];
                        } else {
                            $gameSystem._shopStocks[stockKey] = -1;
                        }
                    }
                    this._data.push(item);
                    this._price.push(goods[2] === 0 ? item.price : goods[3]);
                }
            }
        }

        goodsToItem(goods) {
            switch (goods[0]) {
                case 0: return $dataItems[goods[1]];
                case 1: return $dataWeapons[goods[1]];
                case 2: return $dataArmors[goods[1]];
                default: return null;
            }
        }

        drawItem(index) {
            const item = this.itemAt(index);
            const price = this.price(item);
            const rect = this.itemLineRect(index);
            const priceWidth = this.priceWidth();
            const priceX = rect.x + rect.width - priceWidth;
            const nameWidth = rect.width - priceWidth;

            this.changePaintOpacity(this.isEnabled(item));
            this.drawItemName(item, rect.x, rect.y, nameWidth);
            this.drawText(price, priceX, rect.y, priceWidth, "right");

            const stockKey = this.getStockKey(item);
            const stock = $gameSystem._shopStocks?.[stockKey];
            const stockText = (stock === -1) ? "在庫:∞" : `在庫:${stock}`;
            this.drawText(stockText, priceX - 128, rect.y, 128, "left");

            this.changePaintOpacity(true);
        }

        priceWidth() {
            return 96;
        }

        setStatusWindow(statusWindow) {
            this._statusWindow = statusWindow;
            this.callUpdateHelp();
        }

        updateHelp() {
            this.setHelpWindowItem(this.item());
            if (this._statusWindow) {
                this._statusWindow.setItem(this.item());
            }
        }
    };





    window.RSTH_IH.Window_ItemList = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this._category = "none";
            this._data = [];
        }

        setCategory(category) {
            if (this._category !== category) {
                this._category = category;
                this.refresh();
                this.scrollTo(0, 0);
            }
        }

        makeItemList() {
            if (this._category === "inventory") {
                const inv = $gameSystem._customInventoryItems || [];
                this._data = inv.map((slot, index) => {
                    if (slot) {
                        slot.__slotIndex = index;
                        return slot;
                    } else {
                        return { __slotIndex: index, empty: true };
                    }
                });
            } else if (this._category === "hotbar") {
                const hot = $gameSystem._customHotbarItems || [];
                this._data = hot.map((slot, index) => {
                    if (slot) {
                        slot.__slotIndex = index;
                        return slot;
                    } else {
                        return { __slotIndex: index, empty: true };
                    }
                });
            } else {
                this._data = [];
            }
        }

        item() {
            return this.itemAt(this.index());
        }

        itemAt(index) {
            return this._data && index >= 0 ? this._data[index] : null;
        }


        drawItem(index) {
            const slot = this.itemAt(index);
            const rect = this.itemLineRect(index);

            if (slot) {
                this.changePaintOpacity(this.isEnabled(slot));

                if (slot.empty) {
                    const text = `スロット${slot.__slotIndex}: 無し`;
                    this.drawText(text, rect.x, rect.y, rect.width, "left");
                } else {
                    const item = this.resolveItem(slot);
                    const price = this.sellingPrice(item);
                    //console.log("price", price);
                    const text = `スロット${slot.__slotIndex}: ${item.name} x${slot.count} : ${price}G`;
                    this.drawText(text, rect.x, rect.y, rect.width, "left");
                }

                this.changePaintOpacity(true);
            }
        }

        resolveItem(slot) {
            switch (slot.type) {
                case "item": return $dataItems[slot.id];
                case "weapon": return $dataWeapons[slot.id];
                case "armor": return $dataArmors[slot.id];
                default: return null;
            }
        }

        sellingPrice(item) {
            //console.log("item.price", item.price);
            return Math.floor((item.price || 0) / 5);
        }

        isEnabled(slot) {
            return !slot.empty && slot.count > 0;
        }


        needsNumber() {
            return false;
        }

        numberWidth() {
            return 0;
        }

        updateHelp() {
            const slot = this.item();
            const item = slot ? this.resolveItem(slot) : null;
            this.setHelpWindowItem(item);
        }

        maxCols() {
            return 1; // 横1列
        }

        colSpacing() {
            return 0;
        }

        maxItems() {
            return this._data ? this._data.length : 1;
        }

        isCurrentItemEnabled() {
            return this.isEnabled(this.item());
        }

        selectLast() {
            this.forceSelect(0);
        }

        refresh() {
            this.makeItemList();
            super.refresh();
        }
    };


    window.RSTH_IH.Window_ItemCategory = class extends Window_HorzCommand {
        constructor(rect) {
            super(rect);
            this._itemWindow = null;
        }

        maxCols() {
            return 3;
        }

        update() {
            super.update();
            if (this._itemWindow) {
                this._itemWindow.setCategory(this.currentSymbol());
            }
        }

        makeCommandList() {
            this.addCommand("インベントリ", "inventory");
            this.addCommand("ホットバー", "hotbar");
        }

        setItemWindow(itemWindow) {
            this._itemWindow = itemWindow;
        }

        needsSelection() {
            return this.maxItems() >= 2;
        }
    };

    window.RSTH_IH.Window_ShopCategory = class extends Window_HorzCommand {
        constructor(rect) {
            super(rect);
        }

        maxCols() {
            return 3; // アイテム・武器・防具
        }

        makeCommandList() {
            this.addCommand("アイテム", "item");
            this.addCommand("武器", "weapon");
            this.addCommand("防具", "armor");
        }

        update() {
            super.update();
            const symbol = this.currentSymbol?.();
            if (symbol) {
                window.RSTH_IH.currentShopCategory = symbol;
            }
        }

    };


    window.RSTH_IH.Window_ShopSell = class extends window.RSTH_IH.Window_ItemList {
        constructor(rect) {
            super(rect);
        }

        isEnabled(slot) {
            return slot && slot.count > 0;
        }
    };

    window.RSTH_IH.Window_ShopNumber = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this._item = null;
            this._max = 1;
            this._price = 0;
            this._number = 1;
            this._currencyUnit = TextManager.currencyUnit;
            this.createButtons();
            this.select(0);
            this._canRepeat = false;
        }

        isScrollEnabled() {
            return false;
        }

        number() {
            return this._number;
        }

        setup(item, max, price) {
            this._item = item;
            this._max = Math.floor(max);
            this._price = price;
            this._number = 1;
            this.placeButtons();
            this.refresh();
        }

        setCurrencyUnit(currencyUnit) {
            this._currencyUnit = currencyUnit;
            this.refresh();
        }

        createButtons() {
            this._buttons = [];
            if (ConfigManager.touchUI) {
                for (const type of ["down2", "down", "up", "up2", "ok"]) {
                    const button = new Sprite_Button(type);
                    this._buttons.push(button);
                    this.addInnerChild(button);
                }
                this._buttons[0].setClickHandler(this.onButtonDown2.bind(this));
                this._buttons[1].setClickHandler(this.onButtonDown.bind(this));
                this._buttons[2].setClickHandler(this.onButtonUp.bind(this));
                this._buttons[3].setClickHandler(this.onButtonUp2.bind(this));
                this._buttons[4].setClickHandler(this.onButtonOk.bind(this));
            }
        }

        placeButtons() {
            const sp = this.buttonSpacing();
            const totalWidth = this.totalButtonWidth();
            let x = (this.innerWidth - totalWidth) / 2;
            for (const button of this._buttons) {
                button.x = x;
                button.y = this.buttonY();
                x += button.width + sp;
            }
        }

        totalButtonWidth() {
            const sp = this.buttonSpacing();
            return this._buttons.reduce((r, button) => r + button.width + sp, -sp);
        }

        buttonSpacing() {
            return 8;
        }

        refresh() {
            super.refresh();
            this.drawItemBackground(0);
            this.drawCurrentItemName();
            this.drawMultiplicationSign();
            this.drawNumber();
            this.drawHorzLine();
            this.drawTotalPrice();
        }

        drawCurrentItemName() {
            const padding = this.itemPadding();
            const x = padding * 2;
            const y = this.itemNameY();
            const width = this.multiplicationSignX() - padding * 3;
            this.drawItemName(this._item, x, y, width);
        }

        drawMultiplicationSign() {
            const sign = this.multiplicationSign();
            const width = this.textWidth(sign);
            const x = this.multiplicationSignX();
            const y = this.itemNameY();
            this.resetTextColor();
            this.drawText(sign, x, y, width);
        }

        multiplicationSign() {
            return "\u00d7";
        }

        multiplicationSignX() {
            const sign = this.multiplicationSign();
            const width = this.textWidth(sign);
            return this.cursorX() - width * 2;
        }

        drawNumber() {
            const x = this.cursorX();
            const y = this.itemNameY();
            const width = this.cursorWidth() - this.itemPadding();
            this.resetTextColor();
            this.drawText(this._number, x, y, width, "right");
        }

        drawHorzLine() {
            const padding = this.itemPadding();
            const lineHeight = this.lineHeight();
            const itemY = this.itemNameY();
            const totalY = this.totalPriceY();
            const x = padding;
            const y = Math.floor((itemY + totalY + lineHeight) / 2);
            const width = this.innerWidth - padding * 2;
            this.drawRect(x, y, width, 5);
        }

        drawTotalPrice() {
            const padding = this.itemPadding();
            const total = this._price * this._number;
            const width = this.innerWidth - padding * 2;
            const y = this.totalPriceY();
            this.drawCurrencyValue(total, this._currencyUnit, 0, y, width);
        }

        itemNameY() {
            return Math.floor(this.innerHeight / 2 - this.lineHeight() * 1.5);
        }

        totalPriceY() {
            return Math.floor(this.itemNameY() + this.lineHeight() * 2);
        }

        buttonY() {
            return Math.floor(this.totalPriceY() + this.lineHeight() * 2);
        }

        cursorWidth() {
            const padding = this.itemPadding();
            const digitWidth = this.textWidth("0");
            return this.maxDigits() * digitWidth + padding * 2;
        }

        cursorX() {
            const padding = this.itemPadding();
            return this.innerWidth - this.cursorWidth() - padding * 2;
        }

        maxDigits() {
            return 2;
        }

        update() {
            super.update();
            this.processNumberChange();
        }

        playOkSound() {
            //
        }

        processNumberChange() {
            if (this.isOpenAndActive()) {
                if (Input.isRepeated("right")) {
                    this.changeNumber(1);
                }
                if (Input.isRepeated("left")) {
                    this.changeNumber(-1);
                }
                if (Input.isRepeated("up")) {
                    this.changeNumber(10);
                }
                if (Input.isRepeated("down")) {
                    this.changeNumber(-10);
                }
            }
        }

        changeNumber(amount) {
            const lastNumber = this._number;
            this._number = (this._number + amount).clamp(1, this._max);
            if (this._number !== lastNumber) {
                this.playCursorSound();
                this.refresh();
            }
        }

        itemRect() {
            const rect = new Rectangle();
            rect.x = this.cursorX();
            rect.y = this.itemNameY();
            rect.width = this.cursorWidth();
            rect.height = this.lineHeight();
            return rect;
        }

        isTouchOkEnabled() {
            return false;
        }

        onButtonUp() {
            this.changeNumber(1);
        }

        onButtonUp2() {
            this.changeNumber(10);
        }

        onButtonDown() {
            this.changeNumber(-1);
        }

        onButtonDown2() {
            this.changeNumber(-10);
        }

        onButtonOk() {
            this.processOk();
        }
    };

    window.RSTH_IH.Window_ShopStatus = class extends Window_StatusBase {
        constructor(rect) {
            super(rect);
            this._item = null;
            this._pageIndex = 0;
            this.refresh();
        }

        refresh() {
            this.contents.clear();

            if (this._item) {
                const category = window.RSTH_IH.currentShopCategory;
                if (category === "weapon") {
                    //console.log("this._item", this._item);
                    //console.log("$gameSystem._customHotbarItems", $gameSystem._customHotbarItems);
                    //console.log("this._item.id", this._item.id);
                    const x = this.itemPadding();
                    const y = 0;
                    // 近接武器
                    if ([10, 11, 31, 32, 109, 110].includes(this._item.id)) {
                        this.drawEquipWeapon(x, y, $gameSystem._customHotbarItems[3]?.name || "なし");
                    }
                    // 遠距離A
                    else if ([13, 14, 15, 16, 17, 18, 19, 20, 21, 50, 51, 52, 53, 54, 55, 56, 57, 58, 90, 91, 92, 93, 94].includes(this._item.id)) {
                        this.drawEquipWeapon(x, y, $gameSystem._customHotbarItems[4]?.name || "なし");
                    }
                    // 遠距離B
                    else if ([43, 67, 68, 69, 72, 38, 39, 40, 41, 42, 73, 74, 75, 76, 77, 113].includes(this._item.id)) {
                        this.drawEquipWeapon(x, y, $gameSystem._customHotbarItems[6]?.name || "なし");
                    }
                    // 特殊
                    else if ([12, 59, 60, 61, 62, 33, 63, 64, 65, 66].includes(this._item.id)) {
                        this.drawEquipWeapon(x, y, $gameSystem._customHotbarItems[5]?.name || "なし");
                    }
                    return;
                }

                const x = this.itemPadding();

                if (this.isEquipItem()) {
                    const y = Math.floor(this.lineHeight() * 1.5);
                    this.drawEquipInfo(x, y);
                } else {
                    this.drawPossession(x, 0);

                }
            }
        }


        setItem(item) {
            this._item = item;
            this.refresh();
        }

        isEquipItem() {
            return DataManager.isWeapon(this._item) || DataManager.isArmor(this._item);
        }

        drawEquipWeapon(x, y, text) {
            if (!text) text = "なし";
            const width = this.innerWidth - this.itemPadding() - x;
            this.changeTextColor(ColorManager.systemColor());
            this.drawText("もっている武器", x, y, width, "right");
            this.resetTextColor();
            this.drawText(text, x, y + this.lineHeight(), width);

        }

        drawPossession(x, y) {
            const width = this.innerWidth - this.itemPadding() - x;
            const possessionWidth = this.textWidth("0000");
            this.changeTextColor(ColorManager.systemColor());
            this.drawText(TextManager.possession, x, y, width - possessionWidth);
            this.resetTextColor();
            this.drawText(window.RSTH_IH.getCustomNumItems(this._item), x, y, width, "right");
        }

        drawEquipInfo(x, y) {
            const members = this.statusMembers();
            for (let i = 0; i < members.length; i++) {
                const actorY = y + Math.floor(this.lineHeight() * i * 2.2);
                this.drawActorEquipInfo(x, actorY, members[i]);
            }
        }

        statusMembers() {
            const start = this._pageIndex * this.pageSize();
            const end = start + this.pageSize();
            return $gameParty.members().slice(start, end);
        }

        pageSize() {
            return 4;
        }

        maxPages() {
            return Math.floor(($gameParty.size() + this.pageSize() - 1) / this.pageSize());
        }

        drawActorEquipInfo(x, y, actor) {
            const item1 = this.currentEquippedItem(actor, this._item.etypeId);
            const width = this.innerWidth - x - this.itemPadding();
            const enabled = actor.canEquip(this._item);
            this.changePaintOpacity(enabled);
            this.resetTextColor();
            this.drawText(actor.name(), x, y, width);
            if (enabled) {
                this.drawActorParamChange(x, y, actor, item1);
            }
            this.drawItemName(item1, x, y + this.lineHeight(), width);
            this.changePaintOpacity(true);
        }

        drawActorParamChange(x, y, actor, item1) {
            const width = this.innerWidth - this.itemPadding() - x;
            const paramId = this.paramId();
            const change = this._item.params[paramId] - (item1 ? item1.params[paramId] : 0);
            this.changeTextColor(ColorManager.paramchangeTextColor(change));
            this.drawText((change > 0 ? "+" : "") + change, x, y, width, "right");
        }

        paramId() {
            return DataManager.isWeapon(this._item) ? 2 : 3;
        }

        currentEquippedItem(actor, etypeId) {
            const list = [];
            const equips = actor.equips();
            const slots = actor.equipSlots();
            for (let i = 0; i < slots.length; i++) {
                if (slots[i] === etypeId) {
                    list.push(equips[i]);
                }
            }
            const paramId = this.paramId();
            let worstParam = Number.MAX_VALUE;
            let worstItem = null;
            for (const item of list) {
                if (item && item.params[paramId] < worstParam) {
                    worstParam = item.params[paramId];
                    worstItem = item;
                }
            }
            return worstItem;
        }

        update() {
            super.update();
            this.updatePage();
        }

        updatePage() {
            if (this.isPageChangeEnabled() && this.isPageChangeRequested()) {
                this.changePage();
            }
        }

        isPageChangeEnabled() {
            return this.visible && this.maxPages() >= 2;
        }

        isPageChangeRequested() {
            return Input.isTriggered("shift") ||
                (TouchInput.isTriggered() && this.isTouchedInsideFrame());
        }

        changePage() {
            this._pageIndex = (this._pageIndex + 1) % this.maxPages();
            this.refresh();
            this.playCursorSound();
        }
    };




    window.RSTH_IH.getCustomNumItems = function (item) {
        const inv = $gameSystem._customInventoryItems || [];
        const hot = $gameSystem._customHotbarItems || [];

        const type = DataManager.isItem(item) ? "item"
            : DataManager.isWeapon(item) ? "weapon"
                : DataManager.isArmor(item) ? "armor" : null;

        let count = 0;

        for (const slot of inv) {
            if (slot && slot.id === item.id && slot.type === type) {
                count += slot.count || 1;
            }
        }

        for (const slot of hot) {
            if (slot && slot.id === item.id && slot.type === type) {
                count += slot.count || 1;
            }
        }

        // ✅ アイテムの場合のみスタックを考慮
        if (type === "item" && window.RSTH_IH.StackSize) {
            return count; // 既にスタック単位で加算されているのでOK
        } else {
            return count;
        }
    };

    // 再帰的に凍結する関数
    function deepFreeze(obj) {
        Object.freeze(obj);
        for (const key in obj) {
            if (
                obj.hasOwnProperty(key) &&
                typeof obj[key] === "object" &&
                obj[key] !== null &&
                !Object.isFrozen(obj[key])
            ) {
                deepFreeze(obj[key]);
            }
        }
        return obj;
    }

    /**
     * ショップで販売する商品テーブル
     * フォーマット: [ [ 商品種別, ID, 価格設定(0デフォ　1右の価格), 価格 ], ... ]
     * 例: [0, 1, 0, 0] はアイテムID1, デフォルト価格
     * 防具の値段　=B2*100+50*(B2-1)
     */
    window.RSTH_IH.ShopGoodsData = deepFreeze({
        basicShop: [
            // アイテム
            [0, 7, 0, 0, 200],
            [0, 8, 0, 0, 100],
            [0, 9, 0, 0, 50],
            [0, 25, 0, 0, 15],
            [0, 26, 0, 0, 15],
            [0, 28, 0, 0, 15],
            [0, 29, 0, 0, 15],
            [0, 30, 0, 0, 15],

            // 武器
            [1, 10, 0, 0, 5],
            [1, 11, 0, 0, 5],
            [1, 13, 0, 0, 9],
            [1, 12, 0, 0, 5],
            [1, 43, 0, 0, 5],
            [1, 73, 0, 0, 5],
            [1, 90, 0, 0, 5],

            // 防具
            [2, 2, 0, 0, 1],
            [2, 3, 0, 0, 1],
            [2, 4, 0, 0, 1],
            [2, 5, 0, 0, 1],
            [2, 6, 0, 0, 1],
            [2, 7, 0, 0, 1],
            [2, 27, 0, 0, 1],
            [2, 28, 0, 0, 1],
            [2, 29, 0, 0, 1],
            [2, 30, 0, 0, 1],
            [2, 46, 0, 0, 1],
            [2, 47, 0, 0, 1],
            [2, 48, 0, 0, 1],
            [2, 49, 0, 0, 1],
            [2, 50, 0, 0, 1],
            [2, 58, 0, 0, 1],
            [2, 59, 0, 0, 1],
            [2, 73, 0, 0, 2],
            [2, 74, 0, 0, 2],
            [2, 75, 0, 0, 2],
            [2, 80, 0, 0, 2],
        ],
        uyuyuShop: [
            // アイテム
            [0, 7, 0, 0, 200],
            [0, 8, 0, 0, 100],
            [0, 9, 0, 0, 50],
            [0, 25, 0, 0, 15],
            [0, 26, 0, 0, 15],
            [0, 28, 0, 0, 15],
            [0, 29, 0, 0, 15],
            [0, 30, 0, 0, 15],

            // 武器
            [1, 31, 0, 0, 5],
            [1, 32, 0, 0, 5],
            [1, 33, 0, 0, 5],
            [1, 38, 0, 0, 5],
            [1, 50, 0, 0, 9],
            [1, 73, 0, 0, 5],
            [1, 90, 0, 0, 5],

            // 防具
            [2, 9, 0, 0, 1],
            [2, 10, 0, 0, 1],
            [2, 11, 0, 0, 1],
            [2, 12, 0, 0, 1],
            [2, 13, 0, 0, 1],
            [2, 33, 0, 0, 1],
            [2, 34, 0, 0, 1],
            [2, 35, 0, 0, 1],
            [2, 36, 0, 0, 1],
            [2, 52, 0, 0, 1],
            [2, 53, 0, 0, 1],
            [2, 54, 0, 0, 1],
            [2, 55, 0, 0, 1],
            [2, 76, 0, 0, 2],
            [2, 77, 0, 0, 2],
            [2, 78, 0, 0, 2],
            [2, 79, 0, 0, 2],
        ],
        pShop: [
            // アイテム
            [0, 7, 0, 0, 200],
            [0, 8, 0, 0, 100],
            [0, 9, 0, 0, 50],
            [0, 25, 0, 0, 15],
            [0, 26, 0, 0, 15],
            [0, 28, 0, 0, 15],
            [0, 29, 0, 0, 15],
            [0, 30, 0, 0, 15],

            // 武器
            [1, 109, 0, 0, 5],
            [1, 110, 0, 0, 5],
            [1, 113, 0, 0, 5],
            [1, 73, 0, 0, 5],
            [1, 90, 0, 0, 5],

            // 防具
            [2, 56, 0, 0, 1],
            [2, 57, 0, 0, 1],
            [2, 60, 0, 0, 1],
            [2, 61, 0, 0, 1],
            [2, 62, 0, 0, 1],
            [2, 63, 0, 0, 1],
            [2, 81, 0, 0, 2],
            [2, 82, 0, 0, 2],
            [2, 83, 0, 0, 2],
            [2, 84, 0, 0, 2],
        ],
        specialShop1: [
            // アイテム
            [0, 7, 1, 50, 200],
            [0, 8, 1, 300, 100],
            [0, 9, 1, 2000, 50],
            [0, 32, 1, 50000, 5],
            [0, 25, 1, 15000, 15],
            [0, 26, 1, 15000, 15],
            [0, 28, 1, 15000, 15],
            [0, 29, 1, 15000, 15],
            [0, 30, 1, 15000, 15],

            // 武器
            [1, 10, 1, 750, 5],
            [1, 11, 1, 3000, 5],
            [1, 13, 1, 2000, 9],
            [1, 12, 1, 1500, 5],
            [1, 43, 1, 500, 5],
            [1, 73, 1, 750, 5],
            [1, 90, 1, 5000, 5],

            // 防具
            [2, 2, 1, 150, 1],
            [2, 3, 1, 300, 1],
            [2, 4, 1, 550, 1],
            [2, 5, 1, 1150, 1],
            [2, 6, 1, 4000, 1],
            [2, 7, 1, 10000, 1],
            [2, 27, 1, 200, 1],
            [2, 28, 1, 600, 1],
            [2, 29, 1, 4500, 1],
            [2, 30, 1, 18000, 1],
            [2, 46, 1, 100, 1],
            [2, 47, 1, 250, 1],
            [2, 48, 1, 500, 1],
            [2, 49, 1, 900, 1],
            [2, 50, 1, 2250, 1],
            [2, 58, 1, 3750, 1],
            [2, 59, 1, 14625, 1],
            [2, 73, 1, 400, 2],
            [2, 74, 1, 1000, 2],
            [2, 75, 1, 3000, 2],
            [2, 80, 1, 13500, 2],
        ],
        specialShop2: [
            // アイテム
            [0, 7, 1, 50, 200],
            [0, 8, 1, 300, 100],
            [0, 9, 1, 2000, 50],
            [0, 32, 1, 50000, 5],
            [0, 25, 1, 15000, 15],
            [0, 26, 1, 15000, 15],
            [0, 28, 1, 15000, 15],
            [0, 29, 1, 15000, 15],
            [0, 30, 1, 15000, 15],

            // 武器
            [1, 31, 1, 1250, 5],
            [1, 32, 1, 6250, 5],
            [1, 33, 1, 2800, 5],
            [1, 38, 1, 750, 5],
            [1, 50, 1, 1950, 9],
            [1, 73, 1, 750, 5],
            [1, 90, 1, 5000, 5],

            // 防具
            [2, 9, 1, 600, 1],
            [2, 10, 1, 1500, 1],
            [2, 11, 1, 6000, 1],
            [2, 12, 1, 13875, 1],
            [2, 13, 1, 29250, 1],
            [2, 33, 1, 300, 1],
            [2, 34, 1, 900, 1],
            [2, 35, 1, 7125, 1],
            [2, 36, 1, 31500, 1],
            [2, 52, 1, 600, 1],
            [2, 53, 1, 1200, 1],
            [2, 54, 1, 6375, 1],
            [2, 55, 1, 48250, 1],
            [2, 76, 1, 1150, 2],
            [2, 77, 1, 2850, 2],
            [2, 78, 1, 7875, 2],
            [2, 79, 1, 15750, 2],
        ],
        specialShop3: [
            // アイテム
            [0, 7, 1, 50, 200],
            [0, 8, 1, 300, 100],
            [0, 9, 1, 2000, 50],
            [0, 32, 1, 50000, 5],
            [0, 25, 1, 15000, 15],
            [0, 26, 1, 15000, 15],
            [0, 28, 1, 15000, 15],
            [0, 29, 1, 15000, 15],
            [0, 30, 1, 15000, 15],

            // 武器
            [1, 109, 1, 750, 5],
            [1, 110, 1, 3000, 5],
            [1, 113, 1, 500, 5],
            [1, 73, 1, 750, 5],
            [1, 90, 1, 5000, 5],

            // 防具
            [2, 56, 1, 600, 1],
            [2, 57, 1, 1200, 1],
            [2, 60, 1, 3000, 1],
            [2, 61, 1, 10875, 1],
            [2, 62, 1, 20250, 1],
            [2, 63, 1, 59500, 1],
            [2, 81, 1, 6300, 2],
            [2, 82, 1, 12000, 2],
            [2, 83, 1, 17700, 2],
            [2, 84, 1, 24325, 2],
        ]
    });


})();