/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {

    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;

    window.RSTH_IH.Scene_SelectMap = class extends Scene_MenuBase {
        create() {
            this.createBackground();
            this.createTitle();
            this.createPortraitLayer();
            super.createWindowLayer();
            this.createConfirmWindow();
            this.createHintText();

            this._loadedMaps = {};
            this.loadAllMaps(() => {
                this.createMapWindows();
                this.onChangeSelect();
            });

            //console.log("ConfigManager.unshoP", ConfigManager.unshoP);
        }

        loadAllMaps(callback) {
            this._mapLoadQueue = [1, 2, 3, 5, 7];
            this._loadNextMap(callback);
        }

        _loadNextMap(callback) {
            if (this._mapLoadQueue.length === 0) {
                callback();
                return;
            }
            const mapId = this._mapLoadQueue.shift();
            DataManager.loadMapData(mapId);
            const check = setInterval(() => {
                if (DataManager.isMapLoaded()) {
                    clearInterval(check);
                    this._loadedMaps[mapId] = JsonEx.makeDeepCopy($dataMap);
                    this._loadNextMap(callback);
                }
            }, 10);
        }

        createHintText() {
            const bitmap = new Bitmap(900, 36);
            const x = 240;

            bitmap.fontSize = 24;
            bitmap.textColor = "#FFFFFF";
            bitmap.drawText("↑↓→←：項目選択", 0, 0, 300, 36, "left");
            bitmap.drawText("Enter：項目決定", x, 0, 300, 36, "left");
            bitmap.drawText("Esc：前の画面、項目へ", x * 2, 0, 300, 36, "left");

            this._HintSprite = new Sprite(bitmap);
            this._HintSprite.x = 120;
            this._HintSprite.y = 6;
            this._HintSprite.visible = true;

            this.addChild(this._HintSprite);
        }

        createBackground() {
            this._backgroundSprite = new window.RSTH_IH.Sprite_CharacterSelectBackground();
            this.addChild(this._backgroundSprite);
        }

        createPortraitLayer() {
            this._portraitSprite = new window.RSTH_IH.Sprite_Map();
            this.addChild(this._portraitSprite);
        }

        createTitle() {
            this._title = new window.RSTH_IH.Sprite_VerticalTitleText("Select Map");
            this.addChild(this._title);
        }

        createMapWindows() {
            const gw = Graphics.boxWidth;
            const gh = Graphics.boxHeight;

            const cm = 12, cw = 96, ch = 96, cs = 16;
            const ctw = cw * 5 + cs * 2;
            const cx = (gw - ctw) / 2;
            const cy = gh - ch - cm * 5;
            const selectRect = new Rectangle(cx, cy, ctw + cm * 5, ch + cm * 5);

            this._selectWindow = new window.RSTH_IH.Window_SelectMap(selectRect, this._loadedMaps);
            this._selectWindow.setHandler("ok", this.onSelectOk.bind(this));
            this._selectWindow.setHandler("cancel", this.popScene.bind(this));
            this._selectWindow.setHandler("change", this.onChangeSelect.bind(this));
            this.addWindow(this._selectWindow);

            const infoRect = new Rectangle(120, 120, 400, gh - 320);
            this._infoWindow = new window.RSTH_IH.Window_MapInfo(infoRect, this._loadedMaps);
            this.addWindow(this._infoWindow);

            this._selectWindow.slideInStart();

        }

        createConfirmWindow() {
            const gw = Graphics.boxWidth;
            const gh = Graphics.boxHeight;
            const rect = new Rectangle((gw - 400) / 2, (gh - 200) / 2, 400, 256);
            this._confirmWindow = new window.RSTH_IH.Window_ConfirmMap("", rect);
            this._confirmWindow.setHandler("yes", this.onConfirmYes.bind(this));
            this._confirmWindow.setHandler("no", this.onConfirmNo.bind(this));
            this._confirmWindow.deactivate();
            this._confirmWindow.hide();
            this.addWindow(this._confirmWindow);
        }

        onChangeSelect() {
            const mapId = this._selectWindow.currentMapId();
            this._infoWindow.setMapId(mapId);
            const map = this._loadedMaps[mapId];
            const picName = map?.meta?.pic;
            if (picName) {
                this._portraitSprite.setPicture(picName);
            }
        }

        onSelectOk() {
            const mapId = this._selectWindow.currentMapId();
            const map = this._loadedMaps[mapId];
            window.RSTH_IH._selectedMapId = mapId;
            //console.log("mapId", mapId);
            this._confirmWindow._mapName = map?.displayName || map?.name || `Map${mapId}`;
            this._confirmWindow.refresh();
            this._selectWindow.deactivate();
            this._confirmWindow.show();
            this._confirmWindow.activate();
        }

        onConfirmYes() {
            // うんしょポイント設定
            const mapId = this._selectWindow.currentMapId();
            const map = this._loadedMaps[mapId];
            window.RSTH_IH.getUnshoP = Number(map.meta.unp);
            window.RSTH_IH.MapName = map.displayName;
            //console.log("window.RSTH_IH.getUnshoP", window.RSTH_IH.getUnshoP);

            //SceneManager.push(window.RSTH_IH.Scene_OptionsCustom);

            window.RSTH_IH.slideScene(this, window.RSTH_IH.Scene_SelectEquipment, "left", 30);
            //window.RSTH_IH.slideScene(this, window.RSTH_IH.Scene_SelectCharacter, "left", 30);
        }


        onConfirmNo() {
            this._confirmWindow.deactivate();
            this._confirmWindow.hide();
            this._selectWindow.activate();
        }

        update() {
            super.update();

            // どの状態でもESCキーで前のシーンに戻る
            if (Input.isTriggered('cancel') && !this._confirmWindow.active) {
                SoundManager.playCancel();
                // this.popScene();
                SceneManager.goto(window.RSTH_IH.Scene_TitleCustom);
            }

            // 確認ウィンドウがアクティブの場合のキャンセルキー
            if (this._confirmWindow.active && Input.isTriggered('cancel')) {
                SoundManager.playCancel();
                this.onConfirmNo();
            }
        }
    };


    window.RSTH_IH.Window_SelectMap = class extends Window_Selectable {
        constructor(rect, loadedMaps) {
            super(rect);
            this._mapIds = [1, 2, 3, 5, 7];
            this._loadedMaps = loadedMaps;
            this.setBackgroundType(2);
            this.refresh();
            this.select(0);
            this.activate();
        }

        maxItems() {
            return 5;
        }

        maxCols() {
            return 5;
        }

        maxRows() {
            return 1;
        }

        itemRect(index) {
            const width = 96 + 12; // 幅を拡張
            const height = 96 + 12; // 高さを拡張
            const spacing = 16;
            const totalWidth = 96 * this.maxCols() + spacing * (this.maxCols() - 1);
            const startX = (this.innerWidth - totalWidth) / 2;
            const x = startX + index * (96 + spacing) - 6; // 左右に余白
            const y = 0 - 6; // 上下に余白
            return new Rectangle(x, y + 6, width, height);
        }

        updateCursor() {
            if (this._cursorAll) {
                this.setCursorRect(0, 0, this.innerWidth, this.innerHeight);
            } else if (this.index() >= 0) {
                const rect = this.itemRect(this.index());
                rect.x -= 2; // 枠をさらに大きく
                rect.y -= 2;
                rect.width += 5;
                rect.height += 5;
                this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
            } else {
                this.setCursorRect(0, 0, 0, 0);
            }
        }

        drawItem(index) {
            const rect = this.itemRect(index);
            const mapId = this._mapIds[index];
            const map = this._loadedMaps[mapId];
            if (!map) return;

            const picName = map.meta?.pic;
            if (!picName) return;

            const bitmap = ImageManager.loadPicture(picName);
            if (!bitmap.isReady()) {
                bitmap.addLoadListener(() => this.refresh());
                return;
            }

            const pw = 96;
            const ph = 96;

            const sx = 0;
            const sy = 0;
            const sw = bitmap.width;
            const sh = bitmap.height;

            const scale = Math.min(pw / sw, ph / sh);
            const dw = sw * scale;
            const dh = sh * scale;

            const dx = rect.x + (rect.width - dw) / 2;
            const dy = rect.y + (rect.height - dh) / 2;

            this.contents.blt(bitmap, sx, sy, sw, sh, dx, dy, dw, dh);
        }

        refresh() {
            this.contents.clear();
            for (let i = 0; i < this.maxItems(); i++) {
                this.drawItem(i);
            }
        }

        currentMapId() {
            return this._mapIds[this.index()];
        }

        slideInStart() {
            this._slideInX = this.x; // 元の位置を保存
            this.x = Graphics.boxWidth; // 画面右端に配置
            this._slideIn = true;
        }

        update() {
            super.update();

            // スライドイン処理
            if (this._slideIn) {
                const speed = 80;
                this.x -= speed;
                if (this.x <= this._slideInX) {
                    this.x = this._slideInX;
                    this._slideIn = false;
                }
            }

            if (this._lastIndex !== this.index()) {
                this._lastIndex = this.index();
                this.callHandler("change");
            }
        }
    };

    window.RSTH_IH.Window_MapInfo = class extends Window_Base {
        constructor(rect, loadedMaps) {
            super(rect);
            this._mapId = null;
            this._loadedMaps = loadedMaps;
        }

        setMapId(mapId) {
            if (this._mapId === mapId) return;
            this._mapId = mapId;
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            const map = this._loadedMaps[this._mapId];
            if (!map) return;

            let y = 12;
            const lineHeight = this.lineHeight();
            const pd = 12;

            //console.log("map", map);
            window.RSTH_IH.MapName = map.displayName;

            this.drawText(map.displayName || map.name || `Map${this._mapId}`, pd, y, this.contents.width, "left");
            y += lineHeight + 12;

            const difficulty = map.meta?.difficulty || "";
            this.drawText("難易度: " + difficulty, pd, y, this.contents.width, "left");
            y += lineHeight + 12;


            window.RSTH_IH.unshoP = ConfigManager.unshoP;
            this.drawText("現在のうんしょポイント: ", pd, y, this.contents.width, "left");
            y += lineHeight + 12;
            this.drawText(window.RSTH_IH.unshoP, pd, y, this.contents.width, "left");
            y += lineHeight + 12;

            const rawProfile = map.meta?.profile || "";
            const lines = rawProfile.replace(/\\n/g, "\n").split("\n");
            for (const line of lines) {
                this.drawText(line, pd, y, this.contents.width, "left");
                y += lineHeight;
            }
        }
    };

    window.RSTH_IH.Sprite_Map = class extends Sprite {
        constructor() {
            super();
            this.anchor.set(0.5, 0.0);
            this.opacity = 0;
            this.scale.set(0.75, 0.75);
            this._currentPic = null;
        }

        setPicture(picName) {
            if (this._currentPic === picName) return;
            this.bitmap = ImageManager.loadPicture(picName);
            this._currentPic = picName;
            this.opacity = 0;
            this.scale.set(0.75, 0.75);
        }

        update() {
            super.update();
            if (!this.bitmap || !this.bitmap.isReady()) return;
            if (this.opacity < 255) this.opacity += 16;
            this.x = Graphics.boxWidth - 420;
            this.y = 64;
        }
    };

    window.RSTH_IH.Sprite_SelectedMapName = class extends Sprite {
        constructor() {
            const w = 480, h = 64;
            const bitmap = new Bitmap(w, h);
            super(bitmap);
            this._name = "";
            this.x = Graphics.boxWidth - w - 20;
            this.y = Graphics.boxHeight - h - 20;
        }

        setName(name) {
            if (this._name === name) return;
            this._name = name;
            this._redraw();
        }

        _redraw() {
            const b = this.bitmap;
            b.clear();
            b.fillRect(0, 0, b.width, b.height, "rgba(0, 0, 0, 0.5)");
            b.fontSize = 36;
            b.textColor = "#ffffff";
            b.outlineColor = "#000000";
            b.outlineWidth = 6;
            b.drawText(this._name, 0, 0, b.width, b.height, "center");
        }
    };

    window.RSTH_IH.Window_ConfirmMap = class extends Window_Command {
        constructor(name, rect) {
            super(rect);
            this._mapName = name;
            this._textLines = 3; // ← 上部に何行テキストを描画するか
            this.refresh();
            this.select(0);
            this.activate();
        }

        makeCommandList() {
            this.addCommand("はい", "yes");
            this.addCommand("いいえ", "no");
        }

        drawAllItems() {
            this.contents.clear();
            const text = `MAPは\n${this._mapName}\nでいいですか？`;
            const lines = text.split("\n");
            const lineHeight = this.lineHeight();

            // 上部テキスト表示
            for (let i = 0; i < lines.length; i++) {
                this.drawText(lines[i], 0, i * lineHeight, this.contentsWidth(), "center");
            }

            // コマンド描画（this.drawItemで描画）
            for (let i = 0; i < this.maxItems(); i++) {
                this.drawItem(i);
            }
        }

        itemRect(index) {
            const rect = super.itemRect(index);
            const offsetY = this.lineHeight() * this._textLines + 12; // ← 上のテキスト行数 + マージン
            rect.y += offsetY;
            return rect;
        }
    };

})();