/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {

    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;

    function clamp(value, min, max) {
        return Math.min(Math.max(value, min), max);
    }

    window.RSTH_IH.Window_BGMSelect = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this._bgmList = window.RSTH_IH.getAvailableMapBGMs?.() || [];
            this._bgmIndex = 0;
            this._lastPlayedIndex = -1;

            const saved = ConfigManager.mapBgmName;
            if (RSTH_DEBUG_LOG) console.log("[saved]ConfigManager.mapBgmName", saved);
            if (saved) {
                const index = this._bgmList.findIndex(bgm => bgm.name === saved);
                if (index >= 0) {
                    this._bgmIndex = index;
                }
            }

            this.refresh();
            this.select(0);
            this.activate();
            this.setHandler("ok", this.processOk.bind(this));
        }


        maxItems() {
            return 1;
        }

        itemHeight() {
            return this.lineHeight();
        }

        cursorRight(wrap) {
            this._bgmIndex = (this._bgmIndex + 1) % this._bgmList.length;
            this.refresh();
            this.playPreview();
        }

        cursorLeft(wrap) {
            this._bgmIndex = (this._bgmIndex - 1 + this._bgmList.length) % this._bgmList.length;
            this.refresh();
            this.playPreview();
        }

        playPreview() {
            if (this._bgmIndex !== this._lastPlayedIndex) {
                const bgm = this._bgmList[this._bgmIndex];
                AudioManager.playBgm({
                    name: bgm.name,
                    volume: bgm.volume ?? 20,
                    pitch: 100,
                    pan: 0
                });
                this._lastPlayedIndex = this._bgmIndex;
            }
        }

        currentBgmName() {
            return this._bgmList[this._bgmIndex]?.name || "";
        }

        drawAllItems() {
            this.contents.clear();

            const labelText = `←キーで前　　マップBGM　　→キーで次 `;
            const labelHeight = this.lineHeight();

            // 上にラベルテキスト
            this.drawText(labelText, 0, 6, this.contentsWidth(), "center");

            // 下にBGM名（カーソル対象）
            this.drawItem(0);
        }

        itemRect(index) {
            const rect = super.itemRect(index);
            const labelHeight = this.lineHeight();
            const offsetY = labelHeight + 12; // テキストの高さ + マージン
            rect.y += offsetY;
            return rect;
        }

        drawItem(index) {
            const rect = this.itemRect(index);
            const bgm = this._bgmList[this._bgmIndex];
            const name = (bgm.name || "（なし）");
            this.drawText(name, rect.x, rect.y, rect.width, "center");
        }

        refresh() {
            this.contents.clear();
            this.drawAllItems();
        }

        isOkEnabled() {
            return true;
        }

        processOk() {
            const selected = this._bgmList[this._bgmIndex];
            if (selected) {
                ConfigManager.mapBgmName = selected.name;
                ConfigManager.save();
                if (RSTH_DEBUG_LOG) console.log("selected.name", selected.name);
                if (RSTH_DEBUG_LOG) console.log("ConfigManager.mapBgmName", ConfigManager.mapBgmName);
            }
            SoundManager.playOk();
            this.callOkHandler();
        }



        processTouch() {
            // 何もしない。クリック無効化
        }

        update() {
            super.update();
            this.contentsOpacity = this.active ? 255 : 96;
        }

    };

    window.RSTH_IH.Window_ConfirmOptions = class extends Window_Command {
        constructor(rect) {
            super(rect);
            this._textLines = 1;
            this.select(0);
            this.activate();
        }

        makeCommandList() {
            this.addCommand("はい", "yes");
            this.addCommand("いいえ", "no");
        }

        drawAllItems() {
            this.contents.clear();
            const text = `この設定でいいですか？`;
            const lineHeight = this.lineHeight();
            const labelY = 6;
            this.drawText(text, 0, labelY, this.contentsWidth(), "center");

            // コマンド描画
            for (let i = 0; i < this.maxItems(); i++) {
                const rect = this.itemRect(i);
                this.drawText(this.commandName(i), rect.x, rect.y, rect.width, "center");
            }
        }

        itemRect(index) {
            const rect = super.itemRect(index);
            const lineHeight = this.lineHeight();
            const offsetY = lineHeight + 12 + 6; // テキスト1行分 + マージン
            rect.y = offsetY + index * lineHeight;
            return rect;
        }

        update() {
            super.update();
            this.contentsOpacity = this.active ? 255 : 96;
        }

    };

    window.RSTH_IH.Scene_OptionsCustom = class extends Scene_MenuBase {
        create() {
            super.create();
            if (RSTH_DEBUG_LOG) console.log(ConfigManager.mapBgmName);
            if (RSTH_DEBUG_LOG) console.log(ConfigManager.lvUpOptionMode);
            if (RSTH_DEBUG_LOG) console.log(ConfigManager.priorityValues);
            if (RSTH_DEBUG_LOG) console.log(ConfigManager.treasureEffectMode);

            //if (RSTH_DEBUG_LOG) console.log("calledFromSelectMap:", this._calledFromSelectMap);

            const gw = Graphics.boxWidth;
            const gh = Graphics.boxHeight;

            // タイトル
            this.createTitle();

            // カテゴリウィンドウ
            const w0 = 360;
            const x0 = 120;
            const y0 = 12 + this.calcWindowHeight(1, true) + 3;
            const y02 = 6;
            const h0 = this.calcWindowHeight(8, true);
            //const h0 = 332;
            const categoryRect = new Rectangle(x0, y0, w0, h0);
            this._categoryWindow = new window.RSTH_IH.Window_CategorySelect(categoryRect);
            this._categoryWindow.setHandler("ok", this.onCategoryOk.bind(this));
            this.addWindow(this._categoryWindow);

            const w1 = 600;
            const w2 = 360;
            const bh = 112;
            const lh = 74;
            const ph = 280;
            const cmh = 160;
            const x1 = gw - w1 - 12;
            const ly = y02 + bh + 12;
            const py = ly + lh + 12;
            const ey = py + ph + 12;
            const cmy = ey + lh + 12;
            const ch = 160;
            const cy = y0 + h0 + 12;
            const apy = cmy + cmh + 12;
            const cgtx = (gw - 400) / 2;
            const cgty = (gh - 200) / 2;
            const cgtw = 400;
            const cgth = this.calcWindowHeight(3, true);;

            // ウィンドウのレイアウト定義
            const layout = {

                bgm: new Rectangle(x1, y02, w1, bh),
                lvup: new Rectangle(x1, ly, w1, lh),
                priority: new Rectangle(x1, py, w1, ph),
                effect: new Rectangle(x1, ey, w1, lh),
                common: new Rectangle(x1, cmy, w1, cmh),
                autopotion: new Rectangle(x1, apy, w1, lh),
                confirm: new Rectangle(x0, cy, w2, ch),
                confirmGotoTitle: new Rectangle(cgtx, cgty, cgtw, cgth),
            };

            // 各ウィンドウ生成
            this._bgmWindow = new window.RSTH_IH.Window_BGMSelect(layout.bgm);
            this._LvUPWindow = new window.RSTH_IH.Window_LvUPOptions(layout.lvup);
            this._prioritySelectWindow = new window.RSTH_IH.Window_PrioritySelect(layout.priority);
            this._treasureWindow = new window.RSTH_IH.Window_TreasureOptions(layout.effect);
            this._optionsWindow = new window.RSTH_IH.Window_OptionsCommon(layout.common);
            this._autoPotionWindow = new window.RSTH_IH.Window_AutoPotion(layout.autopotion);
            this._confirmWindow = new window.RSTH_IH.Window_ConfirmOptions(layout.confirm);
            this._confirmGotoTitleWindow = new window.RSTH_IH.Window_ConfirmGotoTitle(layout.confirmGotoTitle);

            // ハンドラ登録
            this._bgmWindow.setHandler("ok", this.onBgmOk.bind(this));
            this._LvUPWindow.setHandler("ok", this.onLvUpOk.bind(this));
            this._prioritySelectWindow.setHandler("ok", this.onPriorityOk.bind(this));
            this._treasureWindow.setHandler("ok", this.onTreasureOk.bind(this));
            this._optionsWindow.setHandler("ok", this.onOptionsOk.bind(this));
            this._autoPotionWindow.setHandler("ok", this.onAutoPotionOk.bind(this));
            this._confirmWindow.setHandler("yes", this.onConfirmYes.bind(this));
            this._confirmWindow.setHandler("no", this.onConfirmNo.bind(this));

            this._confirmGotoTitleWindow.setHandler("yes", this.onConfirmgtYes.bind(this));
            this._confirmGotoTitleWindow.setHandler("no", this.onConfirmgtNo.bind(this));

            // すべて登録・非表示
            this._windows = [
                this._bgmWindow,
                this._LvUPWindow,
                this._prioritySelectWindow,
                this._treasureWindow,
                this._optionsWindow,
                this._autoPotionWindow,
                this._confirmWindow,
                this._confirmGotoTitleWindow
            ];
            for (const win of this._windows) {
                this.addWindow(win);
                win.deactivate();
                win.hide();
            }

            this.createHintText();

            // 初期状態
            this._categoryWindow.activate();
            this.updateWindowOpacityByCategory(); // ← 初回オパシティ調整

            this._confirmGotoTitleWindow.hide();

            /*
            // BGM初期再生（保存済み or デフォルト1件目）
            const savedBgmName = $gameVariables.value(20);
            const list = this._bgmWindow._bgmList;
            if (list.length > 0) {
                let index = 0;
                if (savedBgmName) {
                    const foundIndex = list.findIndex(bgm => bgm.name === savedBgmName);
                    if (foundIndex >= 0) index = foundIndex;
                }
                this._bgmWindow._bgmIndex = index;
                this._bgmWindow.playPreview();
            }
            */

        }

        initialize() {
            Scene_MenuBase.prototype.initialize.call(this);
            this._calledFromSelectMap = window.RSTH_IH._calledFromSelectMap;
            window.RSTH_IH._calledFromSelectMap = false;
            this._calledFromMapScene = window.RSTH_IH._calledFromMapScene;
            window.RSTH_IH._calledFromMapScene = false;
            this._calledFromTitleScene = window.RSTH_IH._calledFromTitleScene;
            window.RSTH_IH._calledFromTitleScene = false;
        }

        createHintText() {
            const bitmap = new Bitmap(900, 36);
            const x = 240;

            bitmap.fontSize = 24;
            bitmap.textColor = "#FFFFFF";
            bitmap.drawText("↑↓→←：項目選択", 0, 0, 300, 36, "left");
            bitmap.drawText("Enter：項目決定", x, 0, 300, 36, "left");
            bitmap.drawText("Esc：前の画面、項目へ", x * 2, 0, 300, 36, "left");

            this._HintSprite = new Sprite(bitmap);
            this._HintSprite.x = 120;
            this._HintSprite.y = 6;
            this._HintSprite.visible = true;

            this.addChild(this._HintSprite);
        }

        createTitle() {
            this._title = new window.RSTH_IH.Sprite_VerticalTitleText("Option");
            this.addChild(this._title);
        }

        createBackground() {
            this._backgroundSprite = new window.RSTH_IH.Sprite_CharacterSelectBackground();
            this.addChild(this._backgroundSprite); // 最背面に追加
        }

        // カテゴリ決定時
        onCategoryOk() {
            const symbol = this._categoryWindow.currentSymbol();
            this._categoryWindow.deactivate();

            // activate + opacity 更新に統一
            this.updateWindowOpacityByCategory(symbol);

            this._afterCentralOk = (symbol !== "confirm" || symbol !== "gototitle") ? () => {
                this._categoryWindow.activate();
                this.updateWindowOpacityByCategory("none"); // ← 全非アクティブへ
            } : null;

        }


        onBgmOk() {
            const bgmName = this._bgmWindow.currentBgmName();
            ConfigManager.mapBgmName = bgmName;
            ConfigManager.save();

            if (RSTH_DEBUG_LOG) console.log(ConfigManager.mapBgmName);

            this._bgmWindow.deactivate();
            Input.clear();
            this._afterCentralOk?.();
        }

        onLvUpOk() {
            const symbol = this._LvUPWindow.currentSymbol();
            this._LvUPWindow.deactivate();

            // ConfigManager に保存
            ConfigManager.lvUpOptionMode = symbol;
            ConfigManager.save();

            SoundManager.playOk();
            Input.clear();
            this._afterCentralOk?.();
        }

        onPriorityOk() {
            const priorities = this._prioritySelectWindow.getPriorityValues();

            // ConfigManager に保存
            ConfigManager.priorityValues = priorities;
            ConfigManager.save();

            this._prioritySelectWindow.deactivate();
            SoundManager.playOk();
            Input.clear();
            this._afterCentralOk?.();
        }

        onTreasureOk() {
            const symbol = this._treasureWindow.currentSymbol();

            // ConfigManager に保存
            ConfigManager.treasureEffectMode = symbol;
            ConfigManager.save();

            this._treasureWindow.deactivate();
            SoundManager.playOk();
            Input.clear();
            this._afterCentralOk?.();
        }

        onOptionsOk() {
            SoundManager.playOk();
            ConfigManager.save();
            this._optionsWindow.deactivate();
            this._confirmWindow.activate();
            Input.clear();

            this._afterCentralOk?.();
        }

        onAutoPotionOk() {
            SoundManager.playOk();
            ConfigManager.autoPotionPercent = this._autoPotionWindow.getapPerValues()
            ConfigManager.save();

            this._autoPotionWindow.deactivate();
            this._confirmWindow.activate();
            Input.clear();

            //console.log("ConfigManager.autoPotionPercent ", ConfigManager.autoPotionPercent);
            this._afterCentralOk?.();
        }

        onConfirmYes() {
            if (this._calledFromTitleScene) {
                this.returnToTitle();
            } else if (this._calledFromMapScene) {
                this.returnToMap();
            } else {
                //ニューゲーム時
                DataManager.setupNewGame();
                $gameParty._actors = [];
                $gameParty._actors = [window.RSTH_IH._selectedActorId];
                $gamePlayer._actorId = window.RSTH_IH._selectedActorId;
                $gameVariables.setValue(20, window.RSTH_IH._selectedMapId);
                window.RSTH_IH.ProjectileManager._projectiles = [];
                $gameTemp._rsthProjectiles = [];

                //powerupの項目を反映
                const vr = window.RSTH_IH.powerUpValueRates;
                const HPp = ConfigManager["gauge_HP"] ?? 0;
                const Strp = ConfigManager["gauge_Str"] ?? 0;
                const Vitp = ConfigManager["gauge_Vit"] ?? 0;
                const Dexp = ConfigManager["gauge_Dex"] ?? 0;
                const Agip = ConfigManager["gauge_Agi"] ?? 0;
                const Lukp = ConfigManager["gauge_Luk"] ?? 0;
                const Dropp = ConfigManager["gauge_Drop"] ?? 0;
                const Zenyp = ConfigManager["gauge_Zeny"] ?? 0;

                const leader = $gameParty.leader();
                leader._paramPlus[0] += HPp * vr.HP;
                leader._paramPlus[2] += Strp * vr.Str;
                leader._paramPlus[3] += Vitp * vr.Vit;
                leader._paramPlus[5] += Dexp * vr.Dex;
                leader._paramPlus[6] += Agip * vr.Agi;
                leader._paramPlus[7] += Lukp * vr.Luk;

                window.RSTH_IH.PickUpRange = Number(window.RSTH_IH.parameters["pickupRange"] || 3);
                window.RSTH_IH.PickUpRange += Dropp * vr.Drop;

                // クエストイベント用
                window.RSTH_IH.questSpawned = false;
                window.RSTH_IH.BossRush = false;

                window.RSTH_IH.treasureCallFlag = 0;

                window.RSTH_IH._questTextSprite = null;
                window.RSTH_IH._questTextBgSprite = null;
                window.RSTH_IH._questTextDuration = 0;
                window.RSTH_IH._questTextData = null;

                const zeny = Zenyp * vr.Zeny;
                const party = $gameParty;
                party.gainGold(zeny);
                //console.log("leader", leader);

                //window.RSTH_IH._elapsedTimeActive = true;

                window.RSTH_IH.reviveFlag = false;

                window.RSTH_IH._weaponCooldown = 0;
                window.RSTH_IH._skillA_Cooldown = 0;
                window.RSTH_IH._skillB_Cooldown = 0;
                window.RSTH_IH._skillC_Cooldown = 0;
                window.RSTH_IH._cooldownMap = {};


                SceneManager.goto(Scene_Map);
            }

        }

        onConfirmNo() {
            this._confirmWindow.deactivate();
            this._categoryWindow.activate();
            this.updateWindowOpacityByCategory("none");
        }

        onConfirmgtYes() {
            this.returnToTitle();
        }

        onConfirmgtNo() {
            this._confirmGotoTitleWindow.deactivate();
            this._categoryWindow.activate();
            this.updateWindowOpacityByCategory("none");
            this._confirmGotoTitleWindow.hide();
        }

        update() {
            super.update();

            if (this._optionsWindow.active && Input.isTriggered("ok")) {
                this._optionsWindow.deactivate();
                this._confirmWindow.activate();
            }

            if (this._calledFromTitleScene && Input.isTriggered("cancel")) {
                SoundManager.playCancel();
                this.returnToTitle();
            } else if (this._calledFromSelectMap && Input.isTriggered("cancel")) {
                SoundManager.playCancel();
                //window.RSTH_IH._elapsedTimeActive = true;
                SceneManager.pop();
            } else if (this._calledFromMapScene && Input.isTriggered("cancel")) {
                SoundManager.playCancel();
                window.RSTH_IH._elapsedTimeActive = true;
                this.returnToMap();
            }
        }

        returnToMap() {
            const name = ConfigManager.mapBgmName;
            if (name) {
                AudioManager.playBgm({ name: name, volume: 20, pitch: 100, pan: 0 });
            }
            SceneManager.goto(Scene_Map);
        }

        returnToTitle() {
            SceneManager.goto(window.RSTH_IH.Scene_TitleCustom);
        }

        updateWindowOpacityByCategory(symbol) {
            const map = {
                bgm: this._bgmWindow,
                lvup: this._LvUPWindow,
                priority: this._prioritySelectWindow,
                effect: this._treasureWindow,
                common: this._optionsWindow,
                autopotion: this._autoPotionWindow,
                confirm: this._confirmWindow,
            };

            for (const key in map) {
                const win = map[key];
                if (symbol === key) {
                    win.opacity = 255;
                    win.contentsOpacity = 255;
                    win.activate();
                    win.select?.(0);
                } else {
                    win.opacity = 160;
                    win.contentsOpacity = 160;
                    win.deactivate();
                }
                win.show(); // 必ず表示
            }
            //console.log("symbol", symbol);
            if (symbol === "gototitle") {
                this._categoryWindow.deactivate();
                this._confirmGotoTitleWindow.show();
                this._confirmGotoTitleWindow.activate();
            }

        }




    };

    window.RSTH_IH.Window_ConfirmGotoTitle = class extends Window_Command {
        constructor(rect) {
            super(rect);
            this._textLines = 1;
            this.select(0);
            this.activate();
        }

        makeCommandList() {
            this.addCommand("はい", "yes");
            this.addCommand("いいえ", "no");
        }

        drawAllItems() {
            this.contents.clear();
            const text = `タイトルに戻りますか？`;
            const lineHeight = this.lineHeight();
            const labelY = 6;
            this.drawText(text, 0, labelY, this.contentsWidth(), "center");

            // コマンド描画
            for (let i = 0; i < this.maxItems(); i++) {
                const rect = this.itemRect(i);
                this.drawText(this.commandName(i), rect.x, rect.y, rect.width, "center");
            }
        }

        itemRect(index) {
            const rect = super.itemRect(index);
            const lineHeight = this.lineHeight();
            const offsetY = lineHeight + 12 + 6; // テキスト1行分 + マージン
            rect.y = offsetY + index * lineHeight;
            return rect;
        }

        update() {
            super.update();
            this.contentsOpacity = this.active ? 255 : 96;
        }

    };

    window.RSTH_IH.getAvailableMapBGMs = function () {
        return [
            { name: "Theme2" },
            { name: "Battle3" },
            { name: "Battle5" },
            { name: "Battle8" },
            { name: "Dungeon1" },
            { name: "Scene1" },
            { name: "Scene7" },
            { name: "Bgm_Fantasy1_01" },
            { name: "Bgm_Fantasy1_29" },
            { name: "RPGM-FantasyResource01" },
            { name: "RPGM-FantasyResource02" },
            { name: "006_stage2" },
            { name: "015_last_stage1" },
            { name: "bgm018_fortress_stage1" },
            { name: "battle-against-the-darkness" },
            { name: "battle-against-the-overlord" },
            { name: "mozetama_07_Bang_Bang_Banbard" },
            { name: "mozetama_09_Surface_Tension" },
            { name: "Battle-LunaticDance_loop" },
            { name: "Battle-BloodyCrescent_loop" },
            { name: "Battle-Dawn_loop" },
            { name: "Battle-Furious-GUITAR_loop" },
            { name: "Battle-Furious-SYNTH_loop" },
            { name: "Battle-Grief_loop" },
            { name: "Battle-SilverMoon_loop" },
            { name: "Final-Battle01_loop" },
            { name: "Under-the-Moonlight_loop" },
            { name: "Cover_Story" },
            { name: "The_Reaper_named_Phoenix" },
            { name: "The_Reaper__named_Phoenix_GD_Remix_" },
            { name: "情動カタルシス" },
            { name: "たぬきちの冒険" },
            { name: "Ucchii0 - The identity【Free Ver】" },
            { name: "Bunnyhop" },
            { name: "【フリーBGM】 かわいい ポップ アップテンポ 明るい ライブ配信 作業用" },
            { name: "魔王魂 ループ  ファンタジー03" },
            { name: "魔王魂 ループ  ファンタジー04" },
            { name: "魔王魂 ループ BGM ネオロック09" },
            { name: "魔王魂 ループ BGM ネオロック13" },
            { name: "魔王魂 ループ BGM ネオロック66" },
            { name: "魔王魂 ループ BGM ネオロック80" },
            { name: "03_Endless_Battle" }
        ];
    };

    window.RSTH_IH.Window_TreasureOptions = class extends Window_Command {
        constructor(rect) {
            super(rect);
            this._textLines = 1;
            this.initSelection();
            this.refresh();
            this.activate();
        }

        makeCommandList() {
            this.addCommand("標準", "nml");
            this.addCommand("簡易", "esy");
        }

        maxScrollY() {
            return 0;
        }

        initSelection() {
            const savedMode = ConfigManager.treasureEffectMode || "nml";
            const index = this.findSymbolIndex(savedMode);
            this.select(index >= 0 ? index : 0);
            this.updateInputData();
        }

        findSymbolIndex(symbol) {
            return this._list.findIndex(cmd => cmd.symbol === symbol);
        }

        currentSymbol() {
            return this.currentData()?.symbol;
        }

        drawAllItems() {
            this.contents.clear();
            const text = `トレジャー獲得演出`;
            const cw = this.contentsWidth();
            this.drawText(text, 6, 6, cw, "left");

            const rect = this.itemRect(0);
            let idx = this.index();
            if (idx < 0) idx = 0;
            const name = this.commandName(idx);
            this.drawText(name, -6, 6, cw, "right");
        }

        itemRect(index) {
            const rect = new Rectangle();
            rect.width = this.contentsWidth();
            rect.height = this.lineHeight();
            rect.x = 0;
            rect.y = 6;
            return rect;
        }

        cursorRight() {
            this.select((this.index() + 1) % this.maxItems());
            this.updateInputData();
            this.refresh();
        }

        cursorLeft() {
            this.select((this.index() - 1 + this.maxItems()) % this.maxItems());
            this.updateInputData();
            this.refresh();
        }

        cursorDown() { }
        cursorUp() { }

        processHandling() {
            if (this.isOpenAndActive()) {
                if (Input.isTriggered("right")) this.cursorRight(); // ← ここを修正
                else if (Input.isTriggered("left")) this.cursorLeft(); // ← ついでに left も統一
                if (Input.isTriggered("ok")) this.processOk();
            }
        }


        processOk() {
            if (this.isCurrentItemEnabled()) {
                this.playOkSound();
                this.updateInputData();
                this.callOkHandler();
            }
        }

        updateInputData() {
            ConfigManager.treasureEffectMode = this.currentSymbol();
            ConfigManager.save();
        }

        update() {
            super.update();
            this.processHandling();
            this.contentsOpacity = this.active ? 255 : 96;
        }
    };


    window.RSTH_IH.Window_LvUPOptions = class extends Window_Command {
        constructor(rect) {
            super(rect);
            this._textLines = 1;
            this.initSelection();
            this.refresh();
            this.activate();
        }

        makeCommandList() {
            this.addCommand("手動", "nml");
            this.addCommand("自動", "ato");
        }

        maxScrollY() {
            return 0;
        }

        initSelection() {
            const savedMode = ConfigManager.lvUpOptionMode || "nml";
            const index = this.findSymbolIndex(savedMode);
            this.select(index >= 0 ? index : 0);
            this.updateInputData();
        }

        findSymbolIndex(symbol) {
            return this._list.findIndex(cmd => cmd.symbol === symbol);
        }

        currentSymbol() {
            return this.currentData()?.symbol;
        }

        drawAllItems() {
            this.contents.clear();
            const text = `LvUP時ステータス加算方法`;
            const cw = this.contentsWidth();
            this.drawText(text, 6, 6, cw, "left");

            const rect = this.itemRect(0);
            let idx = this.index();
            if (idx < 0) idx = 0;
            const name = this.commandName(idx);
            this.drawText(name, -6, 6, cw, "right");
        }

        itemRect(index) {
            const rect = new Rectangle();
            rect.width = this.contentsWidth();
            rect.height = this.lineHeight();
            rect.x = 0;
            rect.y = 6;
            return rect;
        }

        cursorRight() {
            this.select((this.index() + 1) % this.maxItems());
            this.updateInputData();
            this.refresh();
        }

        cursorLeft() {
            this.select((this.index() - 1 + this.maxItems()) % this.maxItems());
            this.updateInputData();
            this.refresh();
        }

        cursorDown() { }
        cursorUp() { }

        processHandling() {
            if (this.isOpenAndActive()) {
                if (Input.isTriggered("right")) this.cursorRight(); // ← ここを修正
                else if (Input.isTriggered("left")) this.cursorLeft(); // ← ついでに left も統一
                if (Input.isTriggered("ok")) this.processOk();
            }
        }

        processOk() {
            if (this.isCurrentItemEnabled()) {
                this.playOkSound();
                this.updateInputData();
                this.callOkHandler();
            }
        }

        updateInputData() {
            ConfigManager.lvUpOptionMode = this.currentSymbol();
            ConfigManager.save();
        }

        update() {
            super.update();
            this.processHandling();
            this.contentsOpacity = this.active ? 255 : 96;
        }
    };

    window.RSTH_IH.Window_PrioritySelect = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this._params = ["Str", "Vit", "Mag", "Dex", "Agi", "Luk"];
            this._priorityLabels = ["最高", "高", "中", "低", "最低"];

            // 修正点：ConfigManager から初期値を取得
            const saved = ConfigManager.priorityValues;
            this._priorities = Array.isArray(saved) && saved.length === 6 ? saved.slice() : window.RSTH_IH.initPriorityVal;

            this.refresh();
            this.select(0);
            this.activate();
        }

        // 以下は変更なし
        maxItems() { return this._params.length; }

        drawAllItems() {
            this.contents.clear();
            const lineHeight = this.lineHeight();
            this.drawText("パラメータ", 6, 0, this.contentsWidth(), "left");
            this.drawText("優先度", -6, 0, this.contentsWidth(), "right");
            for (let i = 0; i < this.maxItems(); i++) {
                this.drawItem(i);
            }
        }

        drawItem(index) {
            const rect = this.itemRect(index);
            const param = this._params[index];
            const priorityIndex = this._priorities[index];
            const priorityLabel = this._priorityLabels[priorityIndex];
            const y = rect.y;
            this.drawText(`${param}:`, rect.x + 16, y, 100, "left");
            this.drawText(priorityLabel, rect.x, y, this.contentsWidth() - 16, "right");
        }

        cursorRight(wrap) {
            const i = this.index();
            this._priorities[i] = Math.min(this._priorities[i] + 1, this._priorityLabels.length - 1);
            this.refresh();
        }

        cursorLeft(wrap) {
            const i = this.index();
            this._priorities[i] = Math.max(this._priorities[i] - 1, 0);
            this.refresh();
        }

        itemHeight() { return this.lineHeight(); }

        itemRect(index) {
            const rect = super.itemRect(index);
            rect.y += this.lineHeight(); // ラベル行の分
            return rect;
        }

        processTouch() { } // マウス無効化

        getPriorityValues() {
            return this._priorities.clone?.() || this._priorities.slice();
        }

        update() {
            super.update();
            this.contentsOpacity = this.active ? 255 : 96;
        }
    };

    window.RSTH_IH.Window_AutoPotion = class extends Window_Selectable {
        constructor(rect) {
            super(rect);

            this._percent = ConfigManager.autoPotionPercent ?? 20;
            this.refresh();
            this.select(0);
            this.activate();
        }

        maxItems() {
            return 1;
        }

        itemHeight() {
            return this.lineHeight();
        }

        itemRect(index) {
            const rect = new Rectangle();
            rect.width = this.contentsWidth();
            rect.height = this.lineHeight();
            rect.x = 0;
            rect.y = 6;
            return rect;
        }

        refresh() {
            this.contents.clear();
            this.drawAllItems();
        }

        drawAllItems() {
            this.contents.clear();
            const text = "自動で回復アイテムを使用するHPの割合";
            const percentText = `${this._percent}%`;
            const y = 6;

            this.drawText(text, 6, y, this.contentsWidth(), "left");
            this.drawText(percentText, -6, y, this.contentsWidth(), "right");
        }

        cursorRight() {
            this._percent = Math.min(this._percent + 5, 100);
            ConfigManager.autoPotionPercent = this._percent;
            SoundManager.playCursor();
            this.refresh();
        }

        cursorLeft() {
            this._percent = Math.max(this._percent - 5, 0);
            ConfigManager.autoPotionPercent = this._percent;
            SoundManager.playCursor();
            this.refresh();
        }

        getapPerValues() {
            return this._percent;
        }

        cursorDown() { }
        cursorUp() { }

        processHandling() {
            Window_Selectable.prototype.processHandling.call(this);
        }

        processOk() {
            if (this.isCurrentItemEnabled()) {
                this.playOkSound();
                this.callOkHandler();
            }
        }

        update() {
            super.update();
            this.processHandling();
            this.contentsOpacity = this.active ? 255 : 96;
        }
    };

    window.RSTH_IH.UseAutoPotion = function () {
        const pottable = [7, 8, 9];
        let potItem = null;
        for (const p of pottable) {
            potItem = $gameSystem._customHotbarItems.find(i => i && i.type === "item" && i.id === p);
            if (potItem) {
                const hotIndex = $gameSystem._customHotbarItems.indexOf(potItem);
                window.RSTH_IH.useInventoryItem(potItem, "hotbar", hotIndex);
                break;
            }

            potItem = $gameSystem._customInventoryItems.find(i => i && i.type === "item" && i.id === p);
            if (potItem) {
                const invIndex = $gameSystem._customInventoryItems.indexOf(potItem);
                window.RSTH_IH.useInventoryItem(potItem, "inventory", invIndex);
                break;
            }
        }

        SceneManager._scene._hotbarWindow.refresh();
        SceneManager._scene._inventoryWindow.refresh();
        if (!potItem || potItem.count <= 0) {
            return;
        }
    };

    // 新規レイアウト
    window.RSTH_IH.Window_CategorySelect = class extends Window_Command {
        constructor(rect) {
            super(rect);
            this.select(0);
            this.activate();
        }

        makeCommandList() {
            this.addCommand("マップBGM", "bgm");
            this.addCommand("LvUP時ステータス加算方法", "lvup");
            this.addCommand("ステータス自動加算の優先度", "priority");
            this.addCommand("トレジャー獲得演出", "effect");
            this.addCommand("音量調整等", "common");
            this.addCommand("自動回復割合", "autopotion");
            this.addCommand("タイトルに戻る", "gototitle");
            this.addCommand("確認して開始", "confirm");
        }

        update() {
            super.update();
            this.contentsOpacity = this.active ? 255 : 96;
        }
    };

    window.RSTH_IH.Window_OptionsCommon = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            /*
            this._options = [
                { name: "常時ダッシュ", symbol: "alwaysDash", type: "toggle" },
                { name: "BGM音量", symbol: "bgmVolume", type: "volume" },
                { name: "BGS音量", symbol: "bgsVolume", type: "volume" },
                { name: "ME音量", symbol: "meVolume", type: "volume" },
                { name: "SE音量", symbol: "seVolume", type: "volume" },
            ];
            */

            this._options = [
                { name: "常時ダッシュ", symbol: "alwaysDash", type: "toggle" },
                { name: "BGM音量", symbol: "bgmVolume", type: "volume" },
                { name: "SE音量", symbol: "seVolume", type: "volume" },
            ];
            this.refresh();
            this.select(0);
            this.activate();
        }

        maxItems() {
            return this._options.length;
        }

        drawItem(index) {
            const option = this._options[index];
            const rect = this.itemRect(index);
            const status = this.statusText(option);
            this.drawText(option.name, rect.x + 6, rect.y, rect.width - 100, "left");
            this.drawText(status, rect.x - 6, rect.y, rect.width, "right");
        }

        statusText(option) {
            const symbol = option.symbol;
            const type = option.type;

            if (type === "toggle") {
                return ConfigManager[symbol] ? "ON" : "OFF";
            } else if (type === "volume") {
                return `${ConfigManager[symbol]}%`;
            }
            return "";
        }

        cursorRight(wrap) {
            this.changeValue(this.index(), 1);
            SoundManager.playCursor();
        }

        cursorLeft(wrap) {
            this.changeValue(this.index(), -1);
            SoundManager.playCursor();
        }

        changeValue(index, sign) {
            const option = this._options[index];
            const symbol = option.symbol;

            if (option.type === "toggle") {
                ConfigManager[symbol] = !ConfigManager[symbol];
            } else if (option.type === "volume") {
                const current = ConfigManager[symbol];
                const value = Math.max(0, Math.min(current + sign * 10, 100));
                ConfigManager[symbol] = value;
                if (symbol === "seVolume") {
                    ConfigManager.bgsVolume = value;
                    ConfigManager.meVolume = value;

                }
            }

            this.refresh();
        }

        processOk() {
            SoundManager.playOk();
            this.callOkHandler();
        }

        refresh() {
            this.contents.clear();
            this.drawAllItems();
        }

        processHandling() {
            Window_Selectable.prototype.processHandling.call(this);
        }

        processTouch() {
            // 何もしない
        }

        update() {
            super.update();
            this.contentsOpacity = this.active ? 255 : 96;
        }
    };

})();