/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {

    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;

    window.RSTH_IH.Window_SelectCharacter = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this._actorIds = [1, 2, 3];
            this.setBackgroundType(2);
            this.refresh();
            this.select(0);
            this.activate();
        }

        maxItems() {
            return 3;
        }

        maxCols() {
            return 3;
        }

        maxRows() {
            return 1;
        }

        itemRect(index) {
            const width = 96 + 12; // 幅を拡張
            const height = 96 + 12; // 高さを拡張
            const spacing = 16;
            const totalWidth = 96 * this.maxCols() + spacing * (this.maxCols() - 1);
            const startX = (this.innerWidth - totalWidth) / 2;
            const x = startX + index * (96 + spacing) - 6; // 左右に余白
            const y = 0 - 6; // 上下に余白
            return new Rectangle(x, y + 6, width, height);
        }

        updateCursor() {
            if (this._cursorAll) {
                this.setCursorRect(0, 0, this.innerWidth, this.innerHeight);
            } else if (this.index() >= 0) {
                const rect = this.itemRect(this.index());
                rect.x -= 2; // 枠をさらに大きく
                rect.y -= 2;
                rect.width += 4;
                rect.height += 4;
                this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
            } else {
                this.setCursorRect(0, 0, 0, 0);
            }
        }

        drawItem(index) {
            const rect = this.itemRect(index);
            const actorId = this._actorIds[index];
            const actor = $dataActors[actorId];
            if (!actor) return;

            const faceName = actor.faceName;
            const faceIndex = actor.faceIndex;

            const bitmap = ImageManager.loadFace(faceName);
            if (!bitmap.isReady()) {
                bitmap.addLoadListener(() => this.refresh());
                return;
            }

            const pw = 96; // 描画先の幅
            const ph = 96; // 描画先の高さ

            const fw = 144; // 顔1個の幅
            const fh = 144; // 顔1個の高さ

            const col = faceIndex % 4;
            const row = Math.floor(faceIndex / 4);
            const sx = col * fw;
            const sy = row * fh;

            const dx = rect.x + (rect.width - pw) / 2;
            const dy = rect.y + (rect.height - ph) / 2;

            this.contents.blt(bitmap, sx, sy, fw, fh, dx, dy, pw, ph);
        }


        refresh() {
            this.contents.clear();
            for (let i = 0; i < this.maxItems(); i++) {
                this.drawItem(i);
            }
        }

        currentActorId() {
            return this._actorIds[this.index()];
        }

        slideInStart() {
            this._slideInX = this.x; // 元の位置を保存
            this.x = Graphics.boxWidth; // 画面右端に配置
            this._slideIn = true;
        }

        update() {
            super.update();

            // スライドイン処理
            if (this._slideIn) {
                const speed = 80;
                this.x -= speed;
                if (this.x <= this._slideInX) {
                    this.x = this._slideInX;
                    this._slideIn = false;
                }
            }


            if (this._lastIndex !== this.index()) {
                this._lastIndex = this.index();
                this.callHandler("change");
            }
        }
    };

    window.RSTH_IH.Window_CharacterInfo = class extends Window_Base {
        constructor(rect) {
            super(rect);
            this._actorId = null;
            this._classIconBitmapLoaded = {};
        }

        setActorId(actorId) {
            if (this._actorId === actorId) return;
            this._actorId = actorId;
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            const actor = $dataActors[this._actorId];
            if (!actor) return;

            let y = 12;
            const lineHeight = this.lineHeight();

            const pd = 12;
            // 名前
            this.drawText(actor.name, pd, y, this.contents.width, "left");
            y += lineHeight + 48;

            // クラス名 + アイコン
            const classId = actor.classId;
            const className = $dataClasses[classId]?.name || "";
            const iconName = actor.meta.classIcon;
            if (iconName) {
                const bitmap = ImageManager.loadSystem(iconName);
                if (!this._classIconBitmapLoaded[iconName]) {
                    this._classIconBitmapLoaded[iconName] = true;
                    bitmap.addLoadListener(() => this.refresh());
                }
                if (bitmap.isReady()) {
                    this.contents.blt(bitmap, -pd, 0, 48, 32, 0, y);
                }
                this.drawText(className, pd + 36 + 8, y, this.contents.width - 36, "left");
            } else {
                this.drawText(className, pd, y, this.contents.width, "left");
            }
            y += 32 + 24;

            // 難易度
            const difficulty = actor.meta.difficulty || "";
            this.drawText("難易度: " + difficulty, pd, y, this.contents.width, "left");
            y += lineHeight + 24;

            // プロフィール
            const rawProfile = actor.meta.profile || "";
            const lines = rawProfile.replace(/\\n/g, "\n").split("\n");
            for (const line of lines) {
                this.drawText(line, pd, y, this.contents.width, "left");
                y += lineHeight;
            }
        }
    };

    window.RSTH_IH.Sprite_Portrait = class extends Sprite {
        constructor() {
            super();
            this.anchor.set(0.5, 0.0); // 左上基準に近い感じで
            this.opacity = 0;
            this.scale.set(1, 1);
            this._currentPic = null;
        }

        setPicture(picName) {
            if (this._currentPic === picName) return;
            this.bitmap = ImageManager.loadPicture(picName);
            this._currentPic = picName;
            this.opacity = 0;
            this.scale.set(1, 1);
        }

        update() {
            super.update();
            if (!this.bitmap || !this.bitmap.isReady()) return;

            // フェードイン + 拡大
            if (this.opacity < 255) this.opacity += 16;
            if (this.scale.x < 1) {
                this.scale.x += 0.02;
                this.scale.y += 0.02;
            }

            // 固定位置に配置（右端 & タイトルの下）
            const marginX = 320;

            const scene = SceneManager._scene;
            if (scene && scene._title) {
                const baseY = scene._title.y;
                this.x = Graphics.boxWidth - marginX;
                this.y = 0;
            }
        }
    };



    window.RSTH_IH.Scene_SelectCharacter = class extends Scene_MenuBase {
        create() {
            this.createBackground();
            this.createTitle();
            this.createPortraitLayer();
            this.createActorNameLabel();
            //this.createActorEquipments();

            super.createWindowLayer();

            this.createCharacterWindows();
            this.createConfirmWindow();

            this.createHintText();

            this.onChangeSelect();
        }

        createHintText() {
            const bitmap = new Bitmap(900, 36);
            const x = 240;

            bitmap.fontSize = 24;
            bitmap.textColor = "#FFFFFF";
            bitmap.drawText("↑↓→←：項目選択", 0, 0, 300, 36, "left");
            bitmap.drawText("Enter：項目決定", x, 0, 300, 36, "left");
            bitmap.drawText("Esc：前の画面、項目へ", x * 2, 0, 300, 36, "left");

            this._HintSprite = new Sprite(bitmap);
            this._HintSprite.x = 120;
            this._HintSprite.y = 6;
            this._HintSprite.visible = true;

            this.addChild(this._HintSprite);
        }

        createActorEquipments() {
            this._actorEquipments = new window.RSTH_IH.Sprite_ActorEquipments([
                "☆初期武器：",
                "　【魔剣】ラクリマ Lv.1",
                "　投げナイフ x2",
                "☆初期防具：",
                "　バックラー",
                "　レザーキャップ",
                "　レザーベスト",
                "☆スキルA：",
                "　バニシングストライク",
                "☆スキルB：",
                "　精霊の加護",
                "☆スキルC：",
                "　薬王の加護",
                "☆初期アイテム：",
                "　低級ポーション x10",
            ]);
            this.addChild(this._actorEquipments);

        }

        createActorNameLabel() {
            this._actorNameLabel = new window.RSTH_IH.Sprite_SelectedActorName();
            this.addChild(this._actorNameLabel);
        }

        createBackground() {
            this._backgroundSprite = new window.RSTH_IH.Sprite_CharacterSelectBackground();
            this.addChild(this._backgroundSprite);
        }

        createPortraitLayer() {
            this._portraitLayer = new Sprite();
            this.addChild(this._portraitLayer);

            this._portraitSprite = new window.RSTH_IH.Sprite_Portrait();
            this._portraitLayer.addChild(this._portraitSprite);
        }

        createTitle() {
            this._title = new window.RSTH_IH.Sprite_VerticalTitleText("Select Character");
            this.addChild(this._title);
        }

        createCharacterWindows() {
            const gw = Graphics.boxWidth;
            const gh = Graphics.boxHeight;

            const cm = 12, cw = 96, ch = 96, cs = 16;
            const ctw = cw * 3 + cs * 2;
            const cx = (gw - ctw) / 2;
            const cy = gh - ch - cm * 3;
            const selectRect = new Rectangle(cx, cy, ctw + cm * 3, ch + cm * 3);

            this._selectWindow = new window.RSTH_IH.Window_SelectCharacter(selectRect);
            this._selectWindow.setHandler("ok", this.onSelectOk.bind(this));
            this._selectWindow.setHandler("cancel", this.popSceneIfAllowed.bind(this));


            this._selectWindow.setHandler("change", this.onChangeSelect.bind(this));
            this.addWindow(this._selectWindow);

            const infoRect = new Rectangle(120, 120, 400, gh - 320);
            this._infoWindow = new window.RSTH_IH.Window_CharacterInfo(infoRect);
            this.addWindow(this._infoWindow);

            this._selectWindow.slideInStart();

        }

        popSceneIfAllowed() {
            if (!this._confirmWindow.active) {
                SoundManager.playCancel();
                this.popScene();
            }
        }

        createConfirmWindow() {
            const gw = Graphics.boxWidth;
            const gh = Graphics.boxHeight;
            const rect = new Rectangle((gw - 400) / 2, (gh - 200) / 2, 400, 256);
            this._confirmWindow = new window.RSTH_IH.Window_ConfirmCharacter("", rect);
            this._confirmWindow.setHandler("yes", this.onConfirmYes.bind(this));
            this._confirmWindow.setHandler("no", this.onConfirmNo.bind(this));
            this._confirmWindow.deactivate();
            this._confirmWindow.hide();
            this.addWindow(this._confirmWindow);
        }

        onChangeSelect() {
            const actorId = this._selectWindow.currentActorId();
            const actor = $dataActors[actorId];
            this._infoWindow.setActorId(actorId);
            const picName = actor?.meta.pic;
            if (picName) {
                this._portraitSprite.setPicture(picName);
            }
            this._actorNameLabel.setName(actor?.name || "");
        }

        onSelectOk() {
            const actorId = this._selectWindow.currentActorId();
            const actor = $dataActors[actorId];
            window.RSTH_IH._selectedActorId = actorId;

            if (window.RSTH_IH._selectedActorId === 1) {
                window.RSTH_IH.SetTreasure = 1;
                window.RSTH_IH.SetShop = 1;
            }
            else if (window.RSTH_IH._selectedActorId === 2) {
                window.RSTH_IH.SetTreasure = 2;
                window.RSTH_IH.SetShop = 2;
            }
            else if (window.RSTH_IH._selectedActorId === 3) {
                window.RSTH_IH.SetTreasure = 3;
                window.RSTH_IH.SetShop = 3;
            }

            this._confirmWindow._actorName = actor?.name || "";
            this._confirmWindow.refresh();
            this._selectWindow.deactivate();
            this._confirmWindow.show();
            this._confirmWindow.activate();
        }

        onConfirmYes() {
            //SceneManager.push(window.RSTH_IH.Scene_SelectMap);

            //window.RSTH_IH._calledFromSelectMap = true;
            window.RSTH_IH.slideScene(this, window.RSTH_IH.Scene_SelectEquipment, "left", 30);
        }

        onConfirmNo() {
            this._confirmWindow.deactivate();
            this._confirmWindow.hide();
            this._selectWindow.activate();
        }

        update() {
            super.update();

            if (this._confirmWindow.active && Input.isTriggered('cancel')) {
                SoundManager.playCancel();
                this.onConfirmNo();
            }

        }
    };

    window.RSTH_IH.Sprite_ActorEquipments = class extends Sprite {
        constructor(textLines) {
            const lineCount = Array.isArray(textLines) ? textLines.length : 1;
            const lineHeight = 32;
            const padding = 20;
            const height = lineCount * lineHeight + padding * 2;
            const width = Graphics.width;

            const bitmap = new Bitmap(width, height);
            super(bitmap);
            this._textLines = Array.isArray(textLines) ? textLines : [textLines];
            this._draw();
        }

        _draw() {
            const gw = Graphics.boxWidth;
            const gh = Graphics.boxHeight;

            const b = this.bitmap;
            const padding = 20;
            const lineHeight = 32;

            b.fontSize = 24;
            b.textColor = "#ffffff";
            b.outlineColor = "#000000";
            b.outlineWidth = 4;

            const bgWidth = 400;
            const bgHeight = gh - 320;
            const bgX = 536;
            const bgY = 0;

            b.fillRect(bgX, bgY, bgWidth, bgHeight, "rgba(0, 0, 0, 0.5)");

            for (let i = 0; i < this._textLines.length; i++) {
                const text = this._textLines[i];
                const y = padding + i * lineHeight;
                b.drawText(text, bgX + 6, y, b.width, lineHeight, "left");
            }

            // 配置（中央やや上に）
            this.x = 0;
            this.y = 120;
        }
    };

    window.RSTH_IH.Sprite_VerticalTitleText = class extends Sprite {
        constructor(text) {
            const bitmap = new Bitmap(Graphics.height, 108);
            super(bitmap);
            this._text = text;
            this._draw();
        }

        _draw() {
            const b = this.bitmap;
            b.fontSize = 92;
            b.textColor = "#ffffff";
            b.outlineColor = "#000000";
            b.outlineWidth = 6;
            b.fillRect(0, 0, b.width, b.height, "rgba(0, 0, 0, 0.5)");

            // 文字を描画（Canvasの左から右へ水平に書くが、あとで回転する）
            b.drawText(this._text, 0, 0, b.width, b.height, "center");

            // 回転（時計回り 90度 → 下から上に）
            this.rotation = Math.PI / 2;

            // 配置（左中央に配置）
            this.x = 108;
            this.y = 0;
        }
    };


    window.RSTH_IH.Sprite_SelectedActorName = class extends Sprite {
        constructor() {
            const w = 480, h = 64;
            const bitmap = new Bitmap(w, h);
            super(bitmap);
            this._name = "";
            this.x = Graphics.boxWidth - w - 20;
            this.y = Graphics.boxHeight - h - 20;
        }

        setName(name) {
            if (this._name === name) return;
            this._name = name;
            this._redraw();
        }

        _redraw() {
            const b = this.bitmap;
            b.clear();
            b.fillRect(0, 0, b.width, b.height, "rgba(0, 0, 0, 0.5)");
            b.fontSize = 36;
            b.textColor = "#ffffff";
            b.outlineColor = "#000000";
            b.outlineWidth = 6;
            b.drawText(this._name, 0, 0, b.width, b.height, "center");
        }
    };

    window.RSTH_IH.Window_ConfirmCharacter = class extends Window_Command {
        constructor(name, rect) {
            super(rect);
            this._actorName = name;
            this._textLines = 3; // ← 上部に何行テキストを描画するか
            this.refresh();
            this.select(0);
            this.activate();
        }

        makeCommandList() {
            this.addCommand("はい", "yes");
            this.addCommand("いいえ", "no");
        }

        drawAllItems() {
            this.contents.clear();
            const text = `使用するキャラクターは\n${this._actorName}\nでいいですか？`;
            const lines = text.split("\n");
            const lineHeight = this.lineHeight();

            // 上部テキスト表示
            for (let i = 0; i < lines.length; i++) {
                this.drawText(lines[i], 0, i * lineHeight, this.contentsWidth(), "center");
            }

            // コマンド描画（this.drawItemで描画）
            for (let i = 0; i < this.maxItems(); i++) {
                this.drawItem(i);
            }
        }

        itemRect(index) {
            const rect = super.itemRect(index);
            const offsetY = this.lineHeight() * this._textLines + 12; // ← 上のテキスト行数 + マージン
            rect.y += offsetY;
            return rect;
        }
    };










})();