/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {
    "use strict";

    window.RSTH_IH.Window_LevelUpStatus = class extends Window_Base {
        constructor(rect) {
            super(rect);
            this.setBackgroundType(0);
            this.opacity = 255;
            this.contents.paintOpacity = 255;
            this._from = null;
            this._to = null;
            this._hoverParam = null;
            this._hoverValue = 0;
            this._displayParams = [];
        }

        setHoverParam(paramName, plusValue, baseActor = this._from) {
            this._hoverParam = paramName;
            this._hoverValue = plusValue;
            this._hoverBase = baseActor;
            this.refresh();
        }

        clearHoverParam() {
            this._hoverParam = null;
            this._hoverValue = 0;
            this.refresh();
        }

        setBeforeParams(actorBefore) {
            this._from = actorBefore;
        }

        setAfterParams(actorAfter) {
            this._to = actorAfter;
        }

        setDisplayParams(paramNames) {
            this._displayParams = paramNames;
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            if (!this._from || !this._to) return;

            this.contents.fontSize = 28;
            const spacing = 28;
            const x1 = 12;
            const x2 = 120;
            const x3 = 196;

            const drawRow = (label, fromVal, toVal, y) => {
                this.drawText(label, x1, y + 64, 100, 'left');
                this.drawText(`${fromVal}`, x2, y + 64, 96, 'left');
                if (toVal !== null) {
                    this.drawText(`→ ${toVal}`, x3, y + 64, 128, 'left');
                }
            };

            let y = 0;
            const toLevel = this._from.level + 1;
            drawRow(`Lv.`, this._from.level, toLevel, y);

            y += spacing * 2;

            const baseActor = this._hoverBase || this._from;
            const derived = this._hoverParam
                ? window.RSTH_IH.getDerivedStats(baseActor, this._hoverParam, this._hoverValue)
                : {};

            for (const label of this._displayParams) {
                const fromVal = window.RSTH_IH.getCurrentParamValue(this._from, label);
                const toVal = derived.hasOwnProperty(label) ? derived[label] : null;
                drawRow(label, fromVal, toVal, y);
                y += spacing;
            }
        }
    };

    window.RSTH_IH.getCurrentParamValue = function (actor, name) {
        if (!actor._paramPlus) actor._paramPlus = [];
        const vit = actor.paramBase(3) + actor._paramPlus[3];
        const mdef = Math.round((actor.paramBase(4) + actor._paramPlus[4]) / 2);
        //console.log("actor", actor);
        switch (name) {
            case "MaxHP": return actor.mhp;
            case "MaxSP": return actor.mmp;
            case "Atk": return actor.atk;
            case "MAtk": return actor.mat;
            case "Def": return actor.def;
            case "MDef": return mdef;
            case "Hit": return window.RSTH_IH.calculateHit(actor.param(5), actor.param(7));
            case "Eva": return window.RSTH_IH.calculateEva(actor.param(6), actor.param(7));
            case "Cri": return window.RSTH_IH.calculateCritical(actor.param(5), actor.param(7));
            case "CriEva": return window.RSTH_IH.calculateCriticalEva(actor.param(6), actor.param(7));
            case "AtkSpd": return window.RSTH_IH.calculateAtkSpd(actor.param(6));
            case "Str": return actor.param(2);
            case "Vit": return vit;
            case "Mag": return actor.param(4);
            case "Dex": return actor.param(5);
            case "Agi": return actor.param(6);
            case "Luk": return actor.param(7);
            default: return 0;
        }
    };

    window.RSTH_IH.getDerivedStats = function (actor, paramName, value) {
        const result = {};
        if (!actor._paramPlus) actor._paramPlus = [];
        const vit = actor.paramBase(3) + actor._paramPlus[3];
        switch (paramName) {
            case "Str":
                result["Atk"] = actor.param(2) + value;
                result["Str"] = actor.param(2) + value;
                break;
            case "Vit":
                result["MaxHP"] = actor.param(0) + value * 2;
                result["Def"] = actor.param(3) + value;
                result["Vit"] = vit + value;
                break;
            case "Mag":
                result["MaxSP"] = actor.param(1) + value * 2;
                result["MDef"] = Math.round((actor.paramBase(4) + actor._paramPlus[4] + value) / 2);
                result["Mag"] = actor.param(4) + value;
                result["MAtk"] = actor.mat + value;
                break;
            case "Dex":
                const newDex = actor.param(5) + value;
                result["Hit"] = window.RSTH_IH.calculateHit(newDex, actor.param(7));
                result["Cri"] = window.RSTH_IH.calculateCritical(newDex, actor.param(7));
                result["Dex"] = newDex;
                break;
            case "Agi":
                const newAgi = actor.param(6) + value;
                result["Eva"] = window.RSTH_IH.calculateEva(newAgi, actor.param(7));
                result["CriEva"] = window.RSTH_IH.calculateCriticalEva(newAgi, actor.param(7));
                result["AtkSpd"] = window.RSTH_IH.calculateAtkSpd(newAgi);
                result["Agi"] = newAgi;
                break;
            case "Luk":
                const newLuk = actor.param(7) + value;
                result["Hit"] = window.RSTH_IH.calculateHit(actor.param(5), newLuk);
                result["Eva"] = window.RSTH_IH.calculateEva(actor.param(6), newLuk);
                result["Cri"] = window.RSTH_IH.calculateCritical(actor.param(5), newLuk);
                result["CriEva"] = window.RSTH_IH.calculateCriticalEva(actor.param(6), newLuk);
                result["Luk"] = newLuk;
                break;
        }
        return result;
    };

    window.RSTH_IH.Scene_LevelUp = class extends Scene_MenuBase {
        static _before = null;
        static _after = null;

        static setBeforeAndAfter(before, after) {
            this._before = before;
            this._after = after;
        }

        create() {
            super.create();
            const before = window.RSTH_IH.Scene_LevelUp._before;
            const after = window.RSTH_IH.Scene_LevelUp._after;
            this._before = before;
            this._after = after;

            this._title = window.RSTH_IH.createTitleText("レベルアップ！！");
            this.addChild(this._title);

            this._subText = window.RSTH_IH.createSubText(`以下の３つの選択肢から\n上げたいステータスを２つ選択せよ。\n※一回選択したらやり直せないので注意※`);
            this._subText.y = 120;
            this.addChild(this._subText);

            const rectStatus = new Rectangle(100, 120, 360, 700);
            this._statusWindow = new window.RSTH_IH.Window_LevelUpStatus(rectStatus);
            this._statusWindow.setBeforeParams(before);
            this._statusWindow.setAfterParams(after);
            this.addChild(this._statusWindow);

            this._options = window.RSTH_IH.generateRandomOptions();
            this._choiceWindows = [];

            for (let i = 0; i < 3; i++) {
                const rect = new Rectangle(492, 234 + i * 200, 720, 180);
                const win = new window.RSTH_IH.Window_LevelUpChoice(
                    rect,
                    `${this._options[i].name} +${this._options[i].value}`,
                    this._options[i].desc
                );
                win.setHandler("ok", () => this.onChoiceOk(i));
                win.setHandler("cancel", () => { });
                this._choiceWindows.push(win);
            }

            this._selected = [];
        }

        start() {
            super.start();
            for (const win of this._choiceWindows) {
                this.addWindow(win);
                win.refresh();
                win.activate();
                win.select(0);
            }



            this._selected = [];
            this._previewParam = null;
            this._statusWindow.setDisplayParams([
                "MaxHP", "MaxSP", "Atk", "MAtk", "Def", "MDef", "Hit", "Eva", "Cri", "CriEva", "AtkSpd",
                "Str", "Vit", "Mag", "Dex", "Agi", "Luk"
            ]);
        }

        isAnyButtonPressed() {
            return Input.isTriggered('ok') ||
                Input.isTriggered('cancel') ||
                Input.isTriggered('menu') ||
                Input.isTriggered('escape') ||
                TouchInput.isClicked();
        }

        update() {
            if (this.isAnyButtonPressed()) Input.clear();
            super.update();

            const mouseX = TouchInput.x;
            const mouseY = TouchInput.y;
            let previewParam = null;
            let previewValue = 0;

            for (let i = 0; i < this._choiceWindows.length; i++) {
                const win = this._choiceWindows[i];
                const x = win.x;
                const y = win.y;
                const width = win.width;
                const height = win.height;
                const inside = mouseX >= x && mouseX <= x + width &&
                    mouseY >= y && mouseY <= y + height;

                if (win.visible && inside) {
                    previewParam = this._options[i].name;
                    previewValue = this._options[i].value;
                    break;
                }
            }

            const baseActor = this._nextBefore || this._statusWindow._from;
            if (previewParam) {
                this._statusWindow.setHoverParam(previewParam, previewValue, baseActor);
            } else {
                this._statusWindow.clearHoverParam();
            }
        }

        onChoiceOk(index) {
            if (this._selected.includes(index)) return;

            const win = this._choiceWindows[index];
            win.deactivate();
            win.select(-1);
            win.visible = false;
            win.opacity = 160;

            const paramName = this._options[index].name;
            const value = this._options[index].value;
            this._selected.push(index);

            // baseActorは最新のbeforeをコピー（deep）
            const base = JsonEx.parse(JsonEx.stringify(this._nextBefore || this._statusWindow._from));
            const after = JsonEx.parse(JsonEx.stringify(base));

            // Str〜Luk だけを直接加算（派生値はgetDerivedStatsで表示のみに反映）
            const paramIdx = this.paramIndex(paramName);
            if (paramIdx >= 0) {
                after._paramPlus[paramIdx] += value;
            }

            // 🔽 MaxHP/MaxSP も必要に応じて加算
            if (paramName === "Vit") {
                after._paramPlus[0] += value * 2; // MaxHP (paramId 0)
                after._paramPlus[3] += value;   // def
            }
            if (paramName === "Mag") {
                after._paramPlus[1] += value * 2; // MaxSP (paramId 1)
            }

            const actor = $gameParty.leader();
            //console.log("actor", actor);
            // 次のステータス更新
            this._after = after;
            this._nextBefore = JsonEx.parse(JsonEx.stringify(after));

            this._statusWindow.setAfterParams(after);
            this._statusWindow.setBeforeParams(this._nextBefore);
            this._statusWindow.clearHoverParam();

            if (this._selected.length >= 2) {
                SceneManager.pop();
            }
        }

        terminate() {
            super.terminate();

            const actor = $gameParty.leader();

            for (const index of this._selected) {
                const paramName = this._options[index].name;
                const value = this._options[index].value;

                window.RSTH_IH.logLvup(paramName, value);

                switch (paramName) {
                    case "Str": actor.addParam(2, value); break; // Str,Atk
                    case "Vit":
                        actor.addParam(3, value);
                        actor.addParam(0, value * 2);
                        break; // Vit,Def,MaxHP
                    case "Mag":
                        actor.addParam(4, value);
                        actor.addParam(1, value * 2);
                        break; // Matk,Mag,MaxSP
                    case "Dex": actor.addParam(5, value); break; // Dex
                    case "Agi": actor.addParam(6, value); break; // Agi
                    case "Luk": actor.addParam(7, value); break; // Luk
                }
            }
            actor.gainHp(actor.mhp);
            actor.gainMp(actor.mmp);
        }



        paramIndex(name) {
            switch (name) {
                case "MaxHP": return 0;
                case "MaxSP": return 1;
                case "Str": return 2;
                //case "Vit": return 3;
                case "Mag": return 4;
                case "Dex": return 5;
                case "Agi": return 6;
                case "Luk": return 7;
                default: return -1;
            }
        }
    };

    Game_Actor.prototype.displayLevelUp = function (newSkills) {
        // レベルアップ時のメッセージ表示を抑制するため空処理に
    };


    // オーバーライド
    const _Game_Actor_levelUp = Game_Actor.prototype.levelUp;
    Game_Actor.prototype.levelUp = function () {
        const actorBefore = JsonEx.makeDeepCopy(this);

        _Game_Actor_levelUp.call(this);

        const se = {
            name: "Raise3",
            volume: 25,
            pitch: 100,
            pan: 0
        };
        AudioManager.playSe(se);
        window.RSTH_IH.showScreenBorder();

        if (window.RSTH_IH.LvupEasyflagFromPowerUP === 1 || ConfigManager.lvUpOptionMode === "ato") {
            window.RSTH_IH.lvupEasy();
        } else {
            if (SceneManager._scene instanceof Scene_Map) {
                window.RSTH_IH.Scene_LevelUp.setBeforeAndAfter(actorBefore, this);

                SceneManager.push(window.RSTH_IH.Scene_LevelUp);
            }

        }

        if (SceneManager._scene instanceof Scene_Map && SceneManager._scene._statusHUD) {
            SceneManager._scene._statusHUD.refresh();
        }
    };





    // 半透明背景スプライト
    window.RSTH_IH.createDimBackground = function () {
        const sprite = new Sprite();
        sprite.bitmap = new Bitmap(Graphics.width, Graphics.height);
        sprite.bitmap.fillAll("rgba(255,255,255,0.5)");
        return sprite;
    }

    // タイトルテキストスプライト
    window.RSTH_IH.createTitleText = function (text) {
        const sprite = new Sprite(new Bitmap(Graphics.width, 80));
        sprite.bitmap.fontSize = 56;
        sprite.bitmap.drawText(text, 0, 12, Graphics.width, 80, 'center');
        sprite.y = 12;
        return sprite;
    }

    // 説明テキストスプライト
    window.RSTH_IH.createSubText = function (text) {
        const lines = text.split("\n");

        // 🔽 高さに余裕を持たせる（各行24px + 下余白20pxなど）
        const lineHeight = 28;
        const paddingBottom = 16;
        const height = lines.length * lineHeight + paddingBottom;

        const sprite = new Sprite(new Bitmap(Graphics.width, height));
        sprite.bitmap.fontSize = 24;

        for (let i = 0; i < lines.length; i++) {
            // 🔽 i * lineHeight で上下のスペース調整、+24で上部マージン（任意）
            sprite.bitmap.drawText(
                lines[i],
                498,                       // x
                i * lineHeight + 12,      // y（12pxくらいマージン）
                Graphics.width - 498,     // width
                lineHeight,               // height
                'left'
            );
        }

        // 🔽 必要なら sprite.y をここで設定しても良い
        return sprite;
    };


    // ランダムなステータスと値を取得
    window.RSTH_IH.generateRandomOptions = function () {
        const paramList = ["Str", "Vit", "Mag", "Dex", "Agi", "Luk"];
        const descriptions = {
            Str: "【Str（力）】\n物理攻撃力(Atk)が上昇し、\n敵に与えるダメージが増える。",
            Vit: "【Vit（体力）】\n最大HP(MaxHP)と防御力(Def)が上昇し、\n敵からの物理ダメージが減る。",
            Mag: "【Mag（魔力）】\n最大SP(MaxSP)と魔法防御力(MDef)が上昇し、\n敵からの魔法ダメージが減る。\n※現在、効果が未実装のステータス",
            Dex: "【Dex（技巧）】\n命中値(Hit)やクリティカル値(Cri)が上昇し、\n攻撃が当たりやすくなったり、\nクリティカルが少し出やすくなる。",
            Agi: "【Agi（敏捷）】\n回避値(Eva)やクリティカル回避値(CriEva)、\n攻撃速度(AtkSpd)が上昇し、\n敵からの物理攻撃やクリティカル攻撃を\n回避しやすくなる。",
            Luk: "【Luk（運）】\n命中値(Hit)やクリティカル値(Cri)、\n回避値(Eva)やクリティカル回避値(CriEva)\nが上昇する。"
        };

        const shuffled = paramList.sort(() => Math.random() - 0.5);
        const options = [];

        for (let i = 0; i < 3; i++) {
            const name = shuffled[i];
            const value = Math.floor(Math.random() * 3 + 1); // +1〜3
            options.push({
                name,
                value,
                desc: descriptions[name] || ""
            });
        }

        return options;
    };


    // 選択肢ウィンドウ（1つ）
    window.RSTH_IH.Window_LevelUpChoice = class extends Window_Selectable {
        constructor(rect, label, desc) {
            super(rect);
            this._label = label;
            this._desc = desc;
            this.opacity = 255;
            this.contents.paintOpacity = 255;
        }

        itemHeight() {
            return this.height - this.padding * 2;
        }

        isOkEnabled() {
            return true;
        }

        maxItems() { return 1; }

        refresh() {
            this.contents.clear();
            this.contents.fontSize = 24;
            this.drawText(this._label, 12, 6, this.width - 28, 'left');

            const lines = this._desc.split("\n");
            for (let i = 0; i < lines.length; i++) {
                this.drawText(lines[i], 160, i * 28 + 6, this.width - 28, 'left');
            }
        }
    }

    window.RSTH_IH.lvupEasy = function () {
        const params = ["Str", "Vit", "Mag", "Dex", "Agi", "Luk"];
        const options = window.RSTH_IH.generateRandomOptions();
        const actor = $gameParty.leader(); // 例：対象アクター

        const optionsWithPriority = options.map(opt => {
            const paramIndex = params.indexOf(opt.name);
            const priority = ConfigManager.priorityValues[paramIndex] ?? 4;
            const currentValue = window.RSTH_IH.getCurrentParamValue(actor, opt.name);
            return { ...opt, priority, currentValue };
        });

        // 優先度 → value → 成長元パラメータの低さ で並べる
        optionsWithPriority.sort((a, b) => {
            if (a.priority !== b.priority) {
                return a.priority - b.priority; // 優先度昇順
            }
            if (a.value !== b.value) {
                return b.value - a.value;       // value 降順
            }
            return a.currentValue - b.currentValue; // 現在値 昇順（低いほど優先）
        });

        const topTwo = optionsWithPriority.slice(0, 2).map(opt => {
            return { name: opt.name, value: opt.value };
        });

        //console.log("ConfigManager.priorityValues", ConfigManager.priorityValues);
        //console.log("options", options);

        //console.log("topTwo", topTwo); // [{ name: "Mag", value: 3 }, { name: "Dex", value: 4 }] のようになる


        for (const opt of topTwo) {
            const paramName = opt.name;
            const value = opt.value;

            window.RSTH_IH.logLvup(paramName, value);

            switch (paramName) {
                case "Str":
                    actor.addParam(2, value);
                    break;
                case "Vit":
                    actor.addParam(3, value);
                    actor.addParam(0, value * 2);
                    break;
                case "Mag":
                    actor.addParam(4, value);
                    actor.addParam(1, value * 2);
                    break;
                case "Dex":
                    actor.addParam(5, value);
                    break;
                case "Agi":
                    actor.addParam(6, value);
                    break;
                case "Luk":
                    actor.addParam(7, value);
                    break;
            }
        }

        actor.gainHp(actor.mhp);
        actor.gainMp(actor.mmp);

    }

    Game_Actor.prototype.paramBase = function (paramId) {
        const lv = this._level;
        const data = this.currentClass().params[paramId];
        if (data && lv < data.length) {
            return data[lv];
        } else {
            // 例えばLv99以降は最後の値で据え置き
            return data ? data[data.length - 1] : 0;
        }
    };

    Game_Actor.prototype.maxLevel = function () {

        const vr = window.RSTH_IH.powerUpValueRates;
        let MaxLvp = ConfigManager["gauge_MaxLv"] ?? 0;
        MaxLvp = MaxLvp * vr.MaxLv;
        //console.log(" MaxLvp", MaxLvp);
        return MaxLvp + 100;
    };

    Game_Actor.prototype.initExp = function () {
        this._exp = [];
        for (let i = 0; i <= this.maxLevel(); i++) {
            this._exp[i] = this.expForLevel(i);
        }
    };


})();