/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {
    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;

    const DIRS = [
        { dx: 0, dy: -1, dir: 8 }, { dx: 1, dy: 0, dir: 6 },
        { dx: 0, dy: 1, dir: 2 }, { dx: -1, dy: 0, dir: 4 },
        { dx: 1, dy: -1, dir: 9 }, { dx: 1, dy: 1, dir: 3 },
        { dx: -1, dy: 1, dir: 1 }, { dx: -1, dy: -1, dir: 7 }
    ];

    const _Scene_Map_initialize2 = Scene_Map.prototype.initialize;
    Scene_Map.prototype.initialize = function () {
        _Scene_Map_initialize2.call(this);
        if (!window.RSTH_IH._npcManager) {
            window.RSTH_IH._npcManager = new window.RSTH_IH.NpcManager();
        }

    };

    window.RSTH_IH.spawnNpc = function (definitionIndex, x, y) {
        if (!window.RSTH_IH._npcManager) {
            return null;
        }
        return window.RSTH_IH._npcManager.createNpc(definitionIndex, x, y);
    };

    window.RSTH_IH.NpcManager = class {
        constructor() {
            this._definitions = window.RSTH_IH.NpcDefinitions || [];
            this._npcs = {};
            this._npcsOrderedKeys = [];
            this._nextId = 1;
        }

        createNpc(definitionIndex, x, y) {
            const definition = this._definitions[definitionIndex];
            const npcId = "NPC_" + this._nextId++;
            const npc = new window.RSTH_IH.Npc(npcId, definition, x, y);
            this._npcs[npcId] = npc;
            this._npcsOrderedKeys.push(npcId);

            while (this._npcsOrderedKeys.length > window.RSTH_IH.MaxNpcAmount) {
                const oldestId = this._npcsOrderedKeys.shift();
                const npcToRemove = this._npcs[oldestId];
                if (npcToRemove?._databaseData.meta?.normb) {
                    this._npcsOrderedKeys.push(oldestId);
                    continue;
                }
                this.removeNpc(oldestId);
            }

            const tilemap = SceneManager._scene?._spriteset?._tilemap;
            const sprite = npc.sprite?.();
            if (tilemap && sprite && sprite.parent !== tilemap) {
                tilemap.addChild(sprite);
            }

            if (RSTH_DEBUG_LOG) console.log("[spawnNpc ]npc ", npc);
            if (RSTH_DEBUG_LOG) console.log("[spawnNpc ]this._npcs", this._npcs);
            return npc;
        }

        clearAllNpcs() {
            for (const npcId of this._npcsOrderedKeys) {
                this.removeNpc(npcId);
            }

            this._npcs = {};
            if (RSTH_DEBUG_LOG) console.log("[NpcManager] 全Npcを削除しました");
        }

        getAllNpcIdsInOrder() {
            return [...this._npcsOrderedKeys];
        }

        removeNpc(npcId) {
            const npc = this._npcs[npcId];
            if (npc && npc._sprite?.parent) {
                npc._sprite.parent.removeChild(npc._sprite);
            }
            delete this._npcs[npcId];
            const index = this._npcsOrderedKeys.indexOf(npcId);
            if (index !== -1) this._npcsOrderedKeys.splice(index, 1);
        }

        updateAll() {
            const npcs = Object.values(this._npcs);
            for (const npc of npcs) {
                npc.update();
                this.checkPlayerCollision(npc);
            }

            for (const npc of npcs) {
                const sprite = npc._sprite;
                if ((sprite?._isDying || npc._isDead) && !sprite?.parent) {
                    this.removeNpc(npc._id);
                }
            }
        }

        checkPlayerCollision(npc) {
            const px = $gamePlayer.x;
            const py = $gamePlayer.y;
            const dx = Math.abs(npc.x() - px);
            const dy = Math.abs(npc.y() - py);
            if (dx <= 1 && dy <= 1) {
                //console.log(`プレイヤーとNpc(${npc._id})が接近しています`);
            }
        }

        hasNpcAt(x, y, excludeId = null) {
            return Object.values(this._npcs).some(npc => {
                if (excludeId && npc._id === excludeId) return false;
                return npc._x === x && npc._y === y;
            });
        }

        serialize() {
            return Object.values(this._npcs).map(npc => npc.serialize());
        }

        deserialize(dataList) {
            for (const data of dataList) {
                const definition = this._definitions[data.definitionIndex];
                if (!definition) continue;
                const npc = new window.RSTH_IH.Npc(data.id, definition, data.x, data.y);
                npc.deserialize(data);
                this._npcs[data.id] = npc;
                this._npcsOrderedKeys.push(data.id);
            }
        }

        attachAllSprites() {
            const tilemap = SceneManager._scene?._spriteset?._tilemap;
            if (!tilemap) return;
            if (RSTH_DEBUG_LOG) console.log("tilemap", tilemap);
            for (const npc of Object.values(this._npcs)) {
                const sprite = npc.sprite?.();
                if (RSTH_DEBUG_LOG) console.log("sprite", sprite);
                if (sprite && sprite.parent !== tilemap) {
                    tilemap.addChild(sprite);
                }
            }
        }

        detachAllSprites() {
            const tilemap = SceneManager._scene?._spriteset?._tilemap;
            if (!tilemap) return;
            for (const npc of Object.values(this._npcs)) {
                const sprite = npc.sprite?.();
                if (sprite && sprite.parent === tilemap) {
                    tilemap.removeChild(sprite);
                }
            }
        }

        canMoveToSimple(tx, ty, excludeNpcId = null, dir = 2) {
            if (!$gameMap.isValid(tx, ty)) return false;
            if ([1, 7].includes($gameMap.terrainTag(tx, ty))) return false;
            if ($gameMap.eventsXy(tx, ty).length > 0) return false;
            if (this.getNpcAt(tx, ty)) return false;

            if (tx < 0 || ty < 0 || tx >= $gameMap.width() || ty >= $gameMap.height()) return false;
            if ($gamePlayer.x === tx && $gamePlayer.y === ty) return false;
            if (this.hasNpcAt(tx, ty, excludeNpcId)) return false;
            return true;
        }

        getNpcAt(x, y) {
            return Object.values(this._npcs).find(npc => npc._x === x && npc._y === y) || null;
        }
    };

    window.RSTH_IH.Npc = class {
        constructor(npcId, definition, x, y) {
            this._type = "npc";
            this._id = npcId;
            this._definition = definition;
            this._x = x;
            this._y = y;
            this._lv = definition.defaultLv;
            this._exp = definition.defaultExp;
            this._isEnemy = definition.isEnemy;
            this._hitStop = 0;
            this._attackCooldown = 0;
            this._castTime = 0;
            this._castSkill = null;
            this._stateEffectTimes = 0;
            this.pendingHits = [];
            this.pendingHitTimer = 0; // フレームカウント

            this._aiMode = "random";  // 追加: AI行動モード（"random" or "chase"）
            //console.log("this._definition", this._definition);

            const db = this._isEnemy ? $dataEnemies[definition.databaseId] : $dataActors[definition.databaseId];
            //console.log("db", db);
            this._databaseData = db;

            this._traits = db.traits || [];
            this._skills = db.skills || [];

            this._sprite = new window.RSTH_IH.Sprite_Npc(this);
            this._moveCooldown = 1;
            this._direction = 2;
            this._pattern = 1;
            this._animationCount = 0;
            this._realX = x;
            this._realY = y;
            this._moveSpeed = definition.moveSpd;
            this._isDead = false;
            this._states = [];  // ステートIDの配列
            this._boss = false;
            this._nokb = false;
            if (db.meta) {
                if (db.meta.boss) {
                    this._boss = db.meta.boss;
                }
                if (db.meta.nokb) {
                    this._nokb = true;
                }
            }
            //console.log("db.meta", db.meta);
            //console.log("this._boss", this._boss);

            if (db.params) {
                this._maxhp = db.params[0];
                this._hp = db.params[0];
                this._maxmp = db.params[1];
                this._mp = db.params[1];
                this._atk = db.params[2];
                this._def = db.params[3];
                this._matk = db.params[4];
                this._dex = db.params[5];
                this._agi = db.params[6];
                this._luk = db.params[7];
            }
            this._rewardExp = db.exp;
            this._rewardGold = db.gold;

            // Npcクラスのconstructor内（定義取得のあとなど）
            this._weaponChances = [];
            this._weaponChancesOrg = [];
            this._attackDistance = 1; // デフォルト攻撃距離

            const metaDistance = db.note?.match(/<distance:(\d+)>/);
            if (metaDistance) {
                this._attackDistance = Number(metaDistance[1]);
            }

            const note = db.note || "";
            const matches = note.matchAll(/<weaponId:(\d+),(\d+)>/g);
            for (const m of matches) {
                const weaponId = Number(m[1]);
                const chance = Number(m[2]);
                if (!isNaN(weaponId) && !isNaN(chance)) {
                    this._weaponChances.push({ weaponId, chance });
                    this._weaponChancesOrg.push({ weaponId, chance });
                }
            }

            this._castBar = new window.RSTH_IH.Sprite_CastBar(60, 6);
            this._castBar.y = -80; // skill名ポップアップより下に
            this._castBar.visible = false;
            this._sprite.addChild(this._castBar);

            this._target = null;
            this._destX = null;
            this._destY = null;

            this._cart = null;
            this._guards = [];
            this._guardIds = [];

        }

        attemptRetreatFrom(playerX, playerY) {
            const directions = DIRS.slice().sort(() => Math.random() - 0.5); // ランダム退避

            for (const dir of directions) {
                const tx = this._x + dir.dx;
                const ty = this._y + dir.dy;
                if (!window.RSTH_IH.isStandableTile2(tx, ty)) continue;
                if (window.RSTH_IH._npcManager.getNpcAt(tx, ty)) continue;
                if ($gamePlayer.x === tx && $gamePlayer.y === ty) continue;

                this._x = tx;
                this._y = ty;
                this._realX = tx;
                this._realY = ty;
                return true;
            }
            return false;
        }

        queueHit(weapon, knockback, sourceX, sourceY, source) {
            // ノックバックは即時
            const npcManager = window.RSTH_IH._npcManager;

            if (!this._nokb) {
                window.RSTH_IH.applyKnockback(this, sourceX, sourceY, knockback, npcManager);
            }

            // ダメージだけ先に計算して貯める
            const dmg = window.RSTH_IH.damage_calculation(weapon, source, this, "mob", "npc");
            this.pendingHits.push(dmg);

            this.pendingHitTimer = 24; // 0.2秒
        }

        applyPendingHits() {
            if (this.pendingHits.length === 0) return;

            // ダメージ合計
            const totalDamage = this.pendingHits.reduce((sum, v) => sum + v, 0);

            // Pop 表示だけここでまとめて出す
            this.showDamagePopup(totalDamage);

            // リセット
            this.pendingHits = [];

        }

        update() {
            this.updateStateEffectTimers();
            if (this._isDead) {
                if (this._sprite) this._sprite.update();
                return;  // 死亡中はAI・移動処理をスキップ
            }
            if (this.hasState(12)) {// 麻痺
                if (this._sprite) this._sprite.update();  // スプライトは更新し続ける
                return;
            }
            if (this._hitStop > 0) {
                this._hitStop--;
                if (this._sprite) this._sprite.update();  // スプライトは更新し続ける
                return;
            }
            if (this._attackCooldown > 0) {
                this._attackCooldown--;
            }
            if (this._castTime > 0) {
                this._castTime--;

                if (this._castBar) {
                    this._castBar.visible = true;
                    this._castBar.updateProgress(this._castTime);
                }

                if (this._castTime === 0 && this._castSkill) {
                    window.RSTH_IH.showSkillNamePopup(this, this._castSkill);
                    // function (target, source, skill, from = "player")
                    window.RSTH_IH.executeSkill(this._target, this, this._castSkill, "npc");

                    this._castSkill = null;
                    if (this._castBar) this._castBar.visible = false;
                }

                if (this._sprite) this._sprite.update();
                return;
            }

            if (this.pendingHitTimer > 0) {
                this.pendingHitTimer--;
                if (this.pendingHitTimer <= 0) {
                    this.applyPendingHits();
                }
            }

            this.updateAI();
            this.updateMove();
            this.updateAnimation();
            if (this._sprite) this._sprite.update();
        }



        updateAI() {
            if (this.isMoving()) return;
            if (this._moveCooldown > 0) { this._moveCooldown--; return; }

            const offset = this.distancePerFrame(); // ③ キャッシュ
            const playerX = $gamePlayer.x;
            const playerY = $gamePlayer.y;

            if (this._aiMode === "random") {
                const dir = DIRS[Math.floor(Math.random() * DIRS.length)];

                const prevDir = this._direction; // ② 変化検出
                if (dir.dx !== 0 && dir.dy !== 0) {
                    if (Math.abs(dir.dx) >= Math.abs(dir.dy)) {
                        this._direction = (dir.dx > 0) ? 6 : 4;
                    } else {
                        this._direction = (dir.dy > 0) ? 2 : 8;
                    }
                } else {
                    if (dir.dx !== 0) {
                        this._direction = (dir.dx > 0) ? 6 : 4;
                    } else if (dir.dy !== 0) {
                        this._direction = (dir.dy > 0) ? 2 : 8;
                    }
                }
                if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                const tx = this._x + dir.dx;
                const ty = this._y + dir.dy;

                if (this.canMoveTo(tx, ty, dir.dir)) {
                    this._x = tx;
                    this._y = ty;
                    if (this._realX === this._x - dir.dx) this._realX += dir.dx * offset;
                    if (this._realY === this._y - dir.dy) this._realY += dir.dy * offset;
                }
            }

            else if (this._aiMode === "chase") {
                const dx = Math.abs(playerX - this._x); // ④ 共通 dx dy
                const dy = Math.abs(playerY - this._y);
                const dist = dx + dy;
                const attackRange = this._attackDistance ?? 1;

                if (dist <= attackRange && dist > 0) {
                    if (this._attackCooldown <= 0) {
                        this.attackPlayer();
                        this._attackCooldown = this.calculateAtkSpd();
                    }
                } else {
                    const randIndices = [0, 1, 2, 3, 4, 5, 6, 7].sort(() => Math.random() - 0.5); // ⑦ シャッフル代替
                    let bestDir = null;
                    let bestDist = Infinity;

                    for (let i = 0; i < randIndices.length; i++) {
                        const dir = DIRS[randIndices[i]];
                        const tx = this._x + dir.dx;
                        const ty = this._y + dir.dy;
                        if (!this.canMoveTo(tx, ty, dir.dir)) continue;

                        const d = Math.abs(playerX - tx) + Math.abs(playerY - ty);
                        if (d < bestDist) {
                            bestDist = d;
                            bestDir = dir;
                        }
                    }

                    if (bestDir) {
                        const prevDir = this._direction;
                        if (bestDir.dx !== 0 && bestDir.dy !== 0) {
                            if (Math.abs(bestDir.dx) >= Math.abs(bestDir.dy)) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        } else {
                            if (bestDir.dx !== 0) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else if (bestDir.dy !== 0) {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        }
                        if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                        this._x += bestDir.dx;
                        this._y += bestDir.dy;
                        this._realX += bestDir.dx * offset;
                        this._realY += bestDir.dy * offset;
                    }
                }
            }

            else if (this._aiMode === "follow") {
                const playerX = $gamePlayer.x;
                const playerY = $gamePlayer.y;
                const dx = playerX - this._x;
                const dy = playerY - this._y;
                const dist = Math.abs(dx) + Math.abs(dy);


                const followMaxRange = this._followMaxRange ?? 10;

                if (dist > followMaxRange) {
                    const candidates = [];

                    for (const dir of DIRS) {
                        const tx = playerX + dir.dx;
                        const ty = playerY + dir.dy;
                        if ($gamePlayer.x === tx && $gamePlayer.y === ty) continue;
                        if ($gameMap.eventIdXy(tx, ty) === 0) {
                            candidates.push({ x: tx, y: ty });
                        }
                    }

                    if (candidates.length > 0) {
                        AudioManager.playSe({ name: "Heal6", pan: 0, pitch: 100, volume: 5 });
                        const spot = candidates[Math.floor(Math.random() * candidates.length)];
                        this._x = spot.x;
                        this._y = spot.y;
                        this._realX = spot.x;
                        this._realY = spot.y;
                        this._moveCooldown = 30;
                        return;
                    }
                }

                const attackRange = this._attackDistance ?? 1;
                const followRange = 4;

                if (dist > followRange) {
                    const randIndices = [0, 1, 2, 3, 4, 5, 6, 7].sort(() => Math.random() - 0.5); // ⑦ シャッフル代替
                    let bestDir = null;
                    let bestDist = Infinity;

                    for (let i = 0; i < randIndices.length; i++) {
                        const dir = DIRS[randIndices[i]];
                        const tx = this._x + dir.dx;
                        const ty = this._y + dir.dy;
                        if (!this.canMoveTo(tx, ty, dir.dir)) continue;

                        const d = Math.abs(playerX - tx) + Math.abs(playerY - ty);
                        if (d < bestDist) {
                            bestDist = d;
                            bestDir = dir;
                        }
                    }

                    if (bestDir) {
                        const prevDir = this._direction;
                        if (bestDir.dx !== 0 && bestDir.dy !== 0) {
                            if (Math.abs(bestDir.dx) >= Math.abs(bestDir.dy)) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        } else {
                            if (bestDir.dx !== 0) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else if (bestDir.dy !== 0) {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        }
                        if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                        this._x += bestDir.dx;
                        this._y += bestDir.dy;
                        this._realX += bestDir.dx * offset;
                        this._realY += bestDir.dy * offset;
                    }
                } else {
                    if (dist <= attackRange && dist > 0) {
                        if (this._attackCooldown <= 0) {
                            this.attackMob();
                            this._attackCooldown = this.calculateAtkSpd();
                        }
                    }
                }
            }
            else if (this._aiMode === "pass") {
                if (this._destX == null || this._destY == null) return;
                const targetx = this._destX;
                const targety = this._destY;

                const dx = targetx - this._x;
                const dy = targety - this._y;
                const dist = Math.abs(dx) + Math.abs(dy);
                //console.log("this._destX", this._destX);
                //console.log("this._destY", this._destY);

                if (dist > 0) {
                    const randIndices = [0, 1, 2, 3, 4, 5, 6, 7].sort(() => Math.random() - 0.5); // ⑦ シャッフル代替
                    let bestDir = null;
                    let bestDist = Infinity;

                    for (let i = 0; i < randIndices.length; i++) {
                        const dir = DIRS[randIndices[i]];
                        const tx = this._x + dir.dx;
                        const ty = this._y + dir.dy;
                        if (!this.canMoveTo(tx, ty, dir.dir)) continue;

                        const d = Math.abs(targetx - tx) + Math.abs(targety - ty);
                        if (d < bestDist) {
                            bestDist = d;
                            bestDir = dir;
                        }
                    }

                    if (bestDir) {
                        const prevDir = this._direction;
                        if (bestDir.dx !== 0 && bestDir.dy !== 0) {
                            if (Math.abs(bestDir.dx) >= Math.abs(bestDir.dy)) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        } else {
                            if (bestDir.dx !== 0) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else if (bestDir.dy !== 0) {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        }
                        if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                        this._x += bestDir.dx;
                        this._y += bestDir.dy;
                        this._realX += bestDir.dx * offset;
                        this._realY += bestDir.dy * offset;
                    }
                }
                else if (dist === 0) {
                    //console.log("touchaku");
                    window.RSTH_IH.passEventSuccess(3000, 30);

                    window.RSTH_IH._npcManager.removeNpc(this._cart._id);
                    //console.log("this._cart", this._cart);
                    const amount = this._guards.length;

                    for (let i = 0; i < amount; i++) {
                        window.RSTH_IH._npcManager.removeNpc(this._guards[i]._id);
                    }

                    window.RSTH_IH._npcManager.removeNpc(this._id);
                }
            }

            else if (this._aiMode === "cart") {
                //console.log("this._target", this._target);
                if (!this._target) return;
                const targetX = this._target._x;
                const targetY = this._target._y;
                const dx = targetX - this._x;
                const dy = targetY - this._y;
                const dist = Math.abs(dx) + Math.abs(dy);
                //console.log("dist", dist);

                const followMaxRange = 5;

                if (dist > followMaxRange) {
                    const candidates = [];

                    for (const dir of DIRS) {
                        const tx = targetX + dir.dx;
                        const ty = targetY + dir.dy;
                        if (targetX === tx && targetY === ty) continue;
                        if ($gameMap.eventIdXy(tx, ty) === 0) {
                            candidates.push({ x: tx, y: ty });
                        }
                    }

                    if (candidates.length > 0) {
                        const spot = candidates[Math.floor(Math.random() * candidates.length)];
                        this._x = spot.x;
                        this._y = spot.y;
                        this._realX = spot.x;
                        this._realY = spot.y;
                        this._moveCooldown = 30;
                        return;
                    }
                }

                const followRange = 0;

                if (dist > followRange) {
                    const randIndices = [0, 1, 2, 3, 4, 5, 6, 7].sort(() => Math.random() - 0.5); // ⑦ シャッフル代替
                    let bestDir = null;
                    let bestDist = Infinity;

                    for (let i = 0; i < randIndices.length; i++) {
                        const dir = DIRS[randIndices[i]];
                        const tx = this._x + dir.dx;
                        const ty = this._y + dir.dy;
                        if (!this.canMoveTo(tx, ty, dir.dir)) continue;

                        const d = Math.abs(targetX - tx) + Math.abs(targetY - ty);
                        if (d < bestDist) {
                            bestDist = d;
                            bestDir = dir;
                        }
                    }

                    if (bestDir) {
                        const prevDir = this._direction;
                        if (bestDir.dx !== 0 && bestDir.dy !== 0) {
                            if (Math.abs(bestDir.dx) >= Math.abs(bestDir.dy)) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        } else {
                            if (bestDir.dx !== 0) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else if (bestDir.dy !== 0) {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        }
                        if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                        this._x += bestDir.dx;
                        this._y += bestDir.dy;
                        this._realX += bestDir.dx * offset;
                        this._realY += bestDir.dy * offset;
                    }
                }
            }
            else if (this._aiMode === "guard") {
                const targetX = this._cart._x;
                const targetY = this._cart._y;
                const dx = targetX - this._x;
                const dy = targetY - this._y;
                const dist = Math.abs(dx) + Math.abs(dy);

                const followMaxRange = this._followMaxRange ?? 10;

                if (dist > followMaxRange) {
                    const candidates = [];

                    for (const dir of DIRS) {
                        const tx = targetX + dir.dx;
                        const ty = targetY + dir.dy;
                        if (targetX === tx && targetY === ty) continue;
                        if ($gameMap.eventIdXy(tx, ty) === 0) {
                            candidates.push({ x: tx, y: ty });
                        }
                    }

                    if (candidates.length > 0) {
                        const spot = candidates[Math.floor(Math.random() * candidates.length)];
                        this._x = spot.x;
                        this._y = spot.y;
                        this._realX = spot.x;
                        this._realY = spot.y;
                        this._moveCooldown = 30;
                        return;
                    }
                }

                const attackRange = this._attackDistance ?? 1;
                const followRange = 4;

                if (dist > followRange) {
                    const randIndices = [0, 1, 2, 3, 4, 5, 6, 7].sort(() => Math.random() - 0.5); // ⑦ シャッフル代替
                    let bestDir = null;
                    let bestDist = Infinity;

                    for (let i = 0; i < randIndices.length; i++) {
                        const dir = DIRS[randIndices[i]];
                        const tx = this._x + dir.dx;
                        const ty = this._y + dir.dy;
                        if (!this.canMoveTo(tx, ty, dir.dir)) continue;

                        const d = Math.abs(targetX - tx) + Math.abs(targetY - ty);
                        if (d < bestDist) {
                            bestDist = d;
                            bestDir = dir;
                        }
                    }

                    if (bestDir) {
                        const prevDir = this._direction;
                        if (bestDir.dx !== 0 && bestDir.dy !== 0) {
                            if (Math.abs(bestDir.dx) >= Math.abs(bestDir.dy)) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        } else {
                            if (bestDir.dx !== 0) {
                                this._direction = (bestDir.dx > 0) ? 6 : 4;
                            } else if (bestDir.dy !== 0) {
                                this._direction = (bestDir.dy > 0) ? 2 : 8;
                            }
                        }
                        if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                        this._x += bestDir.dx;
                        this._y += bestDir.dy;
                        this._realX += bestDir.dx * offset;
                        this._realY += bestDir.dy * offset;
                    }
                } else {
                    if (dist <= attackRange && dist > 0) {
                        if (this._attackCooldown <= 0) {
                            this.attackMob();
                            this._attackCooldown = this.calculateAtkSpd();
                        }
                    }
                }
            }
            else if (this._aiMode === "shooter") {
                const dx = Math.abs(playerX - this._x); // ④ 共通 dx dy
                const dy = Math.abs(playerY - this._y);
                const dist = dx + dy;
                const attackRange = this._attackDistance ?? 3;

                if (dist <= attackRange && dist > 0) {
                    if (this._attackCooldown <= 0) {
                        if (this._databaseData.id === 63) {
                            this.bossAttack(this._databaseData.id);
                        } else {
                            this.attackPlayer();
                        }
                        this._attackCooldown = this.calculateAtkSpd();
                    }
                }

                const prevDir = this._direction;
                if (dx > dy) {
                    this._direction = (playerX > this._x) ? 6 : 4;
                } else {
                    this._direction = (playerY > this._y) ? 2 : 8;
                }
                if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新
            }

            this._moveCooldown = 1;
        }




        updateMove() {
            let distancePerFrame = 0;
            if (this.hasState(10)) {// 鈍足
                distancePerFrame = 1 / 256;
            } else {
                distancePerFrame = this._moveSpeed / 256;

            }
            const dx = this._x - this._realX;
            const dy = this._y - this._realY;
            if (Math.abs(dx) > 0) this._realX += Math.sign(dx) * Math.min(Math.abs(dx), distancePerFrame);
            if (Math.abs(dy) > 0) this._realY += Math.sign(dy) * Math.min(Math.abs(dy), distancePerFrame);
        }


        isMoving() {
            return (Math.abs(this._realX - this._x) > 0.001 || Math.abs(this._realY - this._y) > 0.001);
        }

        updateAnimation() {
            if (this.isMoving()) {
                this._animationCount++;
                if (this._animationCount >= 15) {
                    this._pattern = (this._pattern % 3) + 1;
                    this._sprite.updatePattern(this._pattern, this._direction);
                    this._animationCount = 0;
                }
            } else {
                this._pattern = 1;
                this._sprite.updatePattern(this._pattern, this._direction);
            }
        }

        canMoveTo(tx, ty, dir) {
            if (tx < 0 || ty < 0 || tx >= $gameMap.width() || ty >= $gameMap.height()) return false;
            if (!$gameMap.isPassable(tx, ty, dir)) return false;
            const events = $gameMap.eventsXy(tx, ty);
            if (events.length > 0) return false;
            if ($gamePlayer.x === tx && $gamePlayer.y === ty) return false;
            if (window.RSTH_IH._npcManager.hasNpcAt(tx, ty, this._id)) return false;
            return true;
        }

        attackPlayer() {
            this._direction = this.calculateDirectionToPlayer();
            this._sprite.updatePattern(this._pattern, this._direction);

            const actions = this._databaseData.actions || [];
            let performedAction = false;

            // 1. まず通常攻撃（skillId:1）を評価
            const normalAttack = actions.find(a => a.skillId === 1);
            if (normalAttack) {
                const rating = normalAttack.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    // 通常攻撃成功
                    this._sprite.startRedFlash();
                    for (const entry of this._weaponChances) {
                        if (Math.random() * 100 < entry.chance) {
                            const weapon = $dataWeapons[entry.weaponId];
                            performedAction = this.useWeapons(weapon);
                        }
                    }

                    if (!performedAction) {
                        // 通常攻撃失敗（確率外）→素手攻撃
                        window.RSTH_IH.damage_calculation(null, this, this, "npc", "actor");
                    }

                    return;
                }
            }

            // 2. 通常攻撃に失敗したら、他のスキルを順に評価
            for (const action of actions) {
                if (action.skillId === 1) continue; // 通常攻撃は除外

                const rating = action.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    const skill = $dataSkills[action.skillId];
                    //(`[NpcSkill] skill`, skill);
                    const castTag = skill.meta?.casttime;
                    const castFrames = castTag ? Number(castTag) : 0;

                    if (castFrames > 0) {
                        this._castTime = castFrames;
                        this._castSkill = skill;


                        if (this._castBar) {
                            this._castBar.setup(castFrames);
                            this._castBar.visible = true;
                        }
                    } else {
                        this._sprite.startRedFlash();
                        //console.log(`[NpcSkill] action`, action);
                        //console.log(`[NpcSkill] Using Skill ID: ${action.skillId}, rating: ${rating}, rand: ${rand}`);
                        window.RSTH_IH.showSkillNamePopup(this, skill);

                        // function (target, source, skill, from = "player")
                        window.RSTH_IH.executeSkill(this._target, this, skill, "npc");
                    }

                    performedAction = true;
                    break;
                }
            }

            // 3. どれも成功しなかった場合は何もしない
            if (!performedAction) {
                //console.log("[NpcSkill] どのスキルも評価失敗 → 行動しない");
            }
        }

        attackMob() {
            const range = this._attackDistance ?? 1;
            const candidates = [];

            for (let dx = -range; dx <= range; dx++) {
                for (let dy = -range; dy <= range; dy++) {
                    if (dx === 0 && dy === 0) continue;
                    const tx = this._x + dx;
                    const ty = this._y + dy;
                    const mob = window.RSTH_IH._mobManager.getMobAt(tx, ty);

                    if (mob && !mob._isDead) {
                        candidates.push({ mob, dx, dy });
                    }
                }
            }

            if (candidates.length === 0) return;

            // 最も近いmobを選択
            candidates.sort((a, b) => {
                const distA = Math.abs(a.dx) + Math.abs(a.dy);
                const distB = Math.abs(b.dx) + Math.abs(b.dy);
                return distA - distB;
            });

            const target = candidates[0];
            const mob = target.mob;
            this._target = target.mob;

            // 向き設定
            if (Math.abs(target.dx) > Math.abs(target.dy)) {
                this._direction = target.dx > 0 ? 6 : 4;
            } else {
                this._direction = target.dy > 0 ? 2 : 8;
            }

            this._sprite.updatePattern(this._pattern, this._direction);

            const actions = this._databaseData.actions || [];
            let performedAction = false;

            // 1. まず通常攻撃（skillId:1）を評価
            const normalAttack = actions.find(a => a.skillId === 1);
            if (normalAttack) {
                const rating = normalAttack.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    // 通常攻撃成功
                    this._sprite.startRedFlash();
                    for (const entry of this._weaponChances) {
                        if (Math.random() * 100 < entry.chance) {
                            const weapon = $dataWeapons[entry.weaponId];
                            performedAction = this.useWeapons(weapon);
                        }
                    }

                    if (!performedAction) {
                        // 通常攻撃失敗（確率外）→素手攻撃
                        // function (item, mob, npc, from = "actor", target = "mob", bowatk = 0)
                        const dmg = window.RSTH_IH.damage_calculation(null, this._target, this, "npc", "mob", 0);
                        mob.showDamagePopup(dmg);
                    }

                    return;
                }
            }

            // 2. 通常攻撃に失敗したら、他のスキルを順に評価
            for (const action of actions) {
                if (action.skillId === 1) continue; // 通常攻撃は除外

                const rating = action.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    const skill = $dataSkills[action.skillId];
                    //(`[NpcSkill] skill`, skill);
                    const castTag = skill.meta?.casttime;
                    const castFrames = castTag ? Number(castTag) : 0;

                    if (castFrames > 0) {
                        this._castTime = castFrames;
                        this._castSkill = skill;


                        if (this._castBar) {
                            this._castBar.setup(castFrames);
                            this._castBar.visible = true;
                        }
                    } else {
                        this._sprite.startRedFlash();
                        //console.log(`[NpcSkill] action`, action);
                        //console.log(`[NpcSkill] Using Skill ID: ${action.skillId}, rating: ${rating}, rand: ${rand}`);
                        window.RSTH_IH.showSkillNamePopup(this, skill);
                        // function (target, source, skill, from = "player")
                        window.RSTH_IH.executeSkill(this._target, this, skill, "npc");
                    }

                    performedAction = true;
                    break;
                }
            }

            // 3. どれも成功しなかった場合は何もしない
            if (!performedAction) {
                //console.log("[NpcSkill] どのスキルも評価失敗 → 行動しない");
            }
        }


        bossAttack(boss_id) {
            this._direction = this.calculateDirectionToPlayer();
            this._sprite.updatePattern(this._pattern, this._direction);

            let weapons = this._weaponChances;

            // weapons が空の場合は武器リストを初期化
            if (weapons.length === 0) {

                // 武器リスト初期化
                this._weaponChances = JSON.parse(JSON.stringify(this._weaponChancesOrg));
                weapons = this._weaponChances;
            }

            // weapons が空でないなら先頭要素を取り出して usedWeapons に追加
            if (weapons.length > 0) {
                this._sprite.startRedFlash();
                const weapon = $dataWeapons[weapons[0].weaponId];
                this.useWeapons(weapon);

                weapons.shift(); // 先頭要素を取り出して削除
            }
        }

        useWeapons(weapon) {
            if (weapon) {
                if (!this._target) return false;

                const isProjectile = !!weapon.meta?.projectile;
                const startX = this._x;
                const startY = this._y;
                const targetX = this._target._x;
                const targetY = this._target._y;
                //console.log("this", this);
                //const targetX = $gamePlayer.x;
                //const targetY = $gamePlayer.y;
                const meta = weapon.meta;
                const atk = this._atk;
                const power = weapon.params[2] || 5;

                if (isProjectile) {
                    if (meta.boomerang) {
                        const maxDistance = Number(meta.maxdistance) || 8;
                        window.RSTH_IH.ProjectileManager.createBoomerang(
                            weapon, startX, startY, targetX, targetY,
                            atk, maxDistance, "npc", this
                        );
                    }
                    else if (meta.knife) {
                        const knifeIds = new Set([43, 67, 68, 69, 72]);
                        const bowIds = new Set([38, 39, 40, 41, 42]);

                        // 投げナイフ
                        if (knifeIds.has(weapon.id)) {
                            const knifeAmounts = { 67: 2, 68: 3, 69: 4, 72: 5 };
                            const amount = knifeAmounts[weapon.id] || 1;
                            for (let i = 0; i < amount; i++) {
                                window.RSTH_IH.ProjectileManager.createKnife(
                                    weapon,
                                    startX, startY,
                                    targetX, targetY,
                                    atk,
                                    15, "npc", this
                                );
                            }
                        }
                        // 天弓
                        else if (bowIds.has(weapon.id)) {
                            const bowAmounts = { 39: 2, 40: 3, 41: 4, 42: 5 };
                            const amount = bowAmounts[weapon.id] || 1;
                            for (let i = 0; i < amount; i++) {
                                window.RSTH_IH.ProjectileManager.createKnife(
                                    weapon,
                                    startX, startY,
                                    targetX, targetY,
                                    atk,
                                    20, "npc", this
                                );
                            }

                        }
                    }
                    // split
                    else if (meta.split) {
                        const splitIds = new Set([73, 74, 75, 76, 77]);
                        if (splitIds.has(weapon.id)) {
                            window.RSTH_IH.ProjectileManager.createSplit(
                                weapon,
                                startX, startY,
                                targetX, targetY,
                                atk,
                                20, "npc", this
                            );
                        }
                    }
                    else if (meta.banana) {
                        const range = 20;
                        window.RSTH_IH.ProjectileManager.createBananaSpread(weapon, power, range, "npc", this);
                    }
                    else {
                        let item = $dataItems[this._databaseData.meta?.arrowId || 40];

                        window.RSTH_IH.ProjectileManager.createArrow(
                            startX, startY, targetX, targetY, item, power, "npc", this
                        );
                    }
                } else {
                    window.RSTH_IH.useWeaponByNpc(this, weapon);
                }
                return true;
            }
            return false;
        }


        addState(stateId) {
            const state = $dataStates[stateId];
            if (!this._states.includes(stateId)) {
                //console.log("stateId", stateId);
                this._states.push(stateId);
                //console.log(`[Npc] ステート ${stateId} を付与`);
                // 必要に応じてステート効果を即時適用
                // 例: パラメータ上昇・ビジュアル変更・状態表示など
            }
            if (!this._stateEffectTimes) this._stateEffectTimes = {};
            let effectTime = 0;
            if (state.meta.et === "ex") {
                effectTime = 9999999;
            } else {
                effectTime = Number(state.meta.et || 0);
            }
            if (effectTime > 0) {
                this._stateEffectTimes[stateId] = effectTime;
                //console.log(`[addState] ステート${state.name} に制限時間 ${effectTime} フレーム設定`);
            }
        }

        removeState(stateId) {
            const index = this._states.indexOf(stateId);
            if (index >= 0) {
                this._states.splice(index, 1);
                //console.log(`[Npc] ステート ${stateId} を解除`);
                // 効果の解除処理もここに追加可能
            }
        }

        hasState(stateId) {
            return this._states.includes(stateId);
        }

        updateStateEffectTimers() {
            if ($gameMessage.isBusy()) return;

            const times = this._stateEffectTimes || {};
            const removed = [];

            for (const stateId of this._states) {
                if (!(stateId in times)) continue;

                times[stateId]--;
                if (times[stateId] <= 0) {
                    removed.push(stateId);
                }
            }

            removed.forEach(stateId => {
                this.removeState(stateId);
                delete times[stateId];
                const state = $dataStates[stateId];
                //console.log(`[updateStateEffectTimers] ${state.name} の効果が切れた`);
            });
        }

        calculateDirectionToPlayer() {
            const dx = $gamePlayer.x - this._x;
            const dy = $gamePlayer.y - this._y;

            if (dx === 0 && dy === 0) {
                return 2; // 同じ位置なら下（保険）
            }

            if (dy > 0) {
                // プレイヤーがnpcより下側にいる場合
                return 2;
            }

            if (dy < 0) {
                // プレイヤーがnpcより上側にいる場合
                return 8; // 真上にいる
            }

            // dy==0 で左右だけにいるとき
            if (dx > 0) return 6;
            if (dx < 0) return 4;

            return 2; // 万が一
        }

        calculateDirectionToTarget(target) {
            // mob npc専用
            //console.log("target", target);
            const dx = target._x - this._x;
            const dy = target._y - this._y;

            if (dx === 0 && dy === 0) {
                return 2; // 同じ位置なら下（保険）
            }

            if (dy > 0) {
                return 2;
            }

            if (dy < 0) {
                return 8; // 真上にいる
            }

            // dy==0 で左右だけにいるとき
            if (dx > 0) return 6;
            if (dx < 0) return 4;

            return 2; // 万が一
        }


        calculateAtkSpd() {
            const minct = 10;
            const aspd = window.RSTH_IH.calculateAtkSpd(this._agi);
            const ct = 120;
            let atkSpd = Math.round(ct - ct * (aspd / 100));
            if (atkSpd < minct) atkSpd = minct;
            return atkSpd;
        }


        distancePerFrame() { return this._moveSpeed / 256; }
        sprite() { return this._sprite; }
        x() { return this._x; }
        y() { return this._y; }

        serialize() {
            return {
                id: this._id, definitionIndex: window.RSTH_IH.NpcDefinitions.indexOf(this._definition),
                x: this._x, y: this._y, lv: this._lv, exp: this._exp
            };
        }

        deserialize(data) { this._lv = data.lv; this._exp = data.exp; }

        takeDamage(amount, cri = 0) {
            if (!this._isDead) {
                this._hp -= amount;
                this._sprite?.startShake();
                if (!this._boss) {
                    if (cri === 1) {
                        this._hitStop = 30;
                    } else {
                        this._hitStop = 10;
                    }
                }
                if (this._hp <= 0) {
                    this._hp = 0;
                    this.die();
                }

            }
        }

        showDamagePopup(amount, cri = 0) {
            if (!this._sprite) return;
            if (!window.RSTH_IH.DmgPopup) return;

            const seId = (cri === 0) ? 10 : 14;
            AudioManager.playSe($dataSystem.sounds[seId]);
            const popup = new window.RSTH_IH.Sprite_DamagePopup(amount, cri);
            this._sprite.addChild(popup);
        }

        setEventGuards(guards) {
            this._guardIds.push(guards._id);

        }

        die() {
            this._isDead = true;
            this._sprite?.startDeathAnimation();

            if (this._aiMode === "cart") {
                const amount = this._guardIds.length;

                for (let i = 0; i < amount; i++) {
                    window.RSTH_IH._npcManager.removeNpc(this._guardIds[i]);
                }
                const text = `X 商人とカートの護衛に失敗した！`;
                window.RSTH_IH.LogWindowManager.addLog(text, 3);
            }
            else if (this._aiMode === "pass") {
                window.RSTH_IH._npcManager.removeNpc(this._cart._id);
                const amount = this._guards.length;

                for (let i = 0; i < amount; i++) {
                    window.RSTH_IH._npcManager.removeNpc(this._guards[i]._id);
                }
                const text = `X 商人とカートの護衛に失敗した！`;
                window.RSTH_IH.LogWindowManager.addLog(text, 3);
            }

            AudioManager.playSe($dataSystem.sounds[11]);
        }
    };

    window.RSTH_IH.Sprite_Npc = class extends Sprite {
        constructor(npc) {
            super();
            this._npc = npc;
            this._loaded = false;
            this._shakeX = 0;
            this._shakeY = 0;
            this._shakePower = 0;
            this.visible = true;

            const definition = npc._definition;
            const bitmap = ImageManager.loadCharacter(definition.characterName);
            this.bitmap = bitmap;

            bitmap.addLoadListener(() => {
                if (window.RSTH_IH.NpcD_mzonoff) {
                    // MZ形式: 8キャラシート
                    this._isBig = (definition.characterName || "").startsWith('$');
                    this._index = definition.characterIndex;
                    this._pw = this._isBig ? bitmap.width / 3 : bitmap.width / 12;
                    this._ph = this._isBig ? bitmap.height / 4 : bitmap.height / 8;
                    this._sxBase = (this._index % (this._isBig ? 1 : 4)) * 3 * this._pw;
                    this._syBase = Math.floor(this._index / (this._isBig ? 1 : 4)) * 4 * this._ph;
                } else {
                    // 1キャラシート: 常に大キャラ扱い
                    this._isBig = true;
                    this._index = 0;
                    this._pw = bitmap.width / 3;   // 3フレーム
                    this._ph = bitmap.height / 4;  // 4方向
                    this._sxBase = 0;
                    this._syBase = 0;
                }

                this.z = 4;
                this._loaded = true;
                this.updatePattern(1, 2);
            });

            this.anchor.x = 0;
            this.anchor.y = 1.45;

            // HPバーはbossのみ
            //console.log("npc", npc);
            //console.log("bitmap", bitmap);

            this._hpBarSprite = new Sprite(new Bitmap(60, 8));
            this.addChild(this._hpBarSprite);
            this._lastHp = -1;

        }

        startShake() { this._shakePower = 25; }
        startRedFlash() { this._redFlashDuration = 10; }

        update() {
            if (!this._loaded) return;
            //if (!this._loaded || !this.bitmap || !this.bitmap.isReady()) return;
            super.update();

            //if (RSTH_DEBUG_LOG) console.log("this._npc", this._npc);
            this.updateHpBar();

            if (this._isDying) {
                this.updateDeathAnimation();
            } else {
                this.updateShake();
                this.updatePosition();

                if (this._redFlashDuration > 0) {
                    this._redFlashDuration--;
                    this.setBlendColor([255, 64, 64, 160]); // 赤フラッシュ
                } else if (this._npc.hasState && this._npc.hasState(12)) {
                    // ✅ ステート12のとき青色に
                    this.setBlendColor([64, 64, 255, 160]);
                } else if (this._npc.hasState && this._npc.hasState(10)) {
                    // ✅ ステート10のとき紫色に
                    this.setBlendColor([150, 32, 150, 160]);
                } else {
                    this.setBlendColor([0, 0, 0, 0]); // 通常
                }
            }
        }

        updateHpBar() {
            const npc = this._npc;
            if (!npc) return;

            if (this._lastHp === npc._hp) return; // HP変化なしなら更新不要

            this._lastHp = npc._hp;

            const bitmap = this._hpBarSprite.bitmap;
            bitmap.clear();

            const maxWidth = 60;
            const hpRate = npc._hp / npc._maxhp;

            // 背景
            bitmap.fillRect(0, 0, maxWidth, 8, "#000000");

            // HPバー（色はプレイヤー用と同じ）
            bitmap.gradientFillRect(
                0, 0,
                Math.floor(maxWidth * hpRate),
                8,
                ColorManager.hpGaugeColor1(),
                ColorManager.hpGaugeColor2()
            );

            // HPバーの位置（キャラクターの上）
            this._hpBarSprite.x = -6; // 幅の半分で中央合わせ
            this._hpBarSprite.y = -8;  // キャラクターの真下に12px程度オフセット

        }

        updateShake() {
            if (this._shakePower > 0) {
                this._shakeX = (Math.random() - 0.5) * this._shakePower;
                this._shakeY = (Math.random() - 0.5) * this._shakePower;
                this._shakePower -= 1;
            } else {
                this._shakeX = 0; this._shakeY = 0;
            }
        }


        updatePosition() {
            const tw = $gameMap.tileWidth();
            const th = $gameMap.tileHeight();
            const dx = $gameMap.displayX();
            const dy = $gameMap.displayY();

            const baseY = this._npc._realY;

            this.x = Math.round((this._npc._realX - dx) * tw + this._shakeX);
            this.y = Math.round((baseY - dy + 1.3) * th + this._shakeY);

            const npcScreenY = baseY;
            const playerY = $gamePlayer.y;
            this.z = (npcScreenY + 0.4 < playerY) ? 3 : 5;
        }

        updatePattern(pattern, direction) {
            if (!this._loaded) return;
            const dirIndex = { 2: 0, 4: 1, 6: 2, 8: 3 }[direction] ?? 0;
            const sx = this._sxBase + (pattern - 1) * this._pw;
            const sy = this._syBase + dirIndex * this._ph;
            this.setFrame(sx, sy, this._pw, this._ph);
        }

        updateDeathAnimation() {
            this._deathFrame++;
            const totalDuration = 60;

            this._offsetX += this._vx / 60;
            this._offsetY += this._vy / 60;
            this._vy += 0.1;

            this.rotation += this._rotationSpeed;

            // フェードアウト処理
            const fadeStart = totalDuration - 20;
            if (this._deathFrame > fadeStart) {
                const fadeProgress = (this._deathFrame - fadeStart) / 20;
                this.opacity = 255 * (1 - fadeProgress);
            }

            // Npc本体座標 + 吹き飛びオフセット
            const tw = $gameMap.tileWidth();
            const th = $gameMap.tileHeight();
            const dx = $gameMap.displayX();
            const dy = $gameMap.displayY();
            this.x = Math.round((this._npc._realX + this._offsetX - dx) * tw);
            this.y = Math.round((this._npc._realY + this._offsetY - dy) * th);

            if (this._deathFrame >= totalDuration) {
                if (this.parent) this.parent.removeChild(this);
                window.RSTH_IH._npcManager.removeNpc(this._npc._id);
            }
        }

        startDeathAnimation() {
            this._isDying = true;
            this._deathFrame = 0;
            this._vx = (Math.random() - 0.5) * 6;
            this._vy = -4 - Math.random() * 10;
            this._rotationSpeed = (Math.random() - 0.5) * 0.5;

            this._offsetX = 0;  // 吹き飛び用オフセットを分離
            this._offsetY = 0;
        }


    };

    // Npc用の武器攻撃関数
    window.RSTH_IH.useWeaponByNpc = function (npc, weapon) {
        if (!weapon) return;

        const playerX = $gamePlayer.x;
        const playerY = $gamePlayer.y;

        const meta = weapon.meta;

        const knockback = Number(meta.knockback) || 0;

        const range = window.RSTH_IH.getWeaponRange(weapon);
        const shape = meta.shape ?? "box";

        for (let dx = -range; dx <= range; dx++) {
            for (let dy = -range; dy <= range; dy++) {
                const tx = playerX + dx;
                const ty = playerY + dy;

                const absDx = Math.abs(dx);
                const absDy = Math.abs(dy);

                const inRange = (shape === "circle") ? (absDx + absDy <= range) : true;

                if (inRange) {
                    if (tx !== playerX || ty !== playerY) continue;
                    if (!window.RSTH_IH.isTileReachable(npc.x(), npc.y(), tx, ty)) continue;

                    if ($gamePlayer.x === tx && $gamePlayer.y === ty) {
                        window.RSTH_IH.applyKnockbackToPlayer(npc, knockback);

                        window.RSTH_IH.damage_calculation(weapon, npc, npc, "npc", "actor");
                    }
                }
            }
        }
    };

    const _Scene_Map_start2 = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
        _Scene_Map_start2.call(this);

        if (!window.RSTH_IH._npcManager) {
            window.RSTH_IH._npcManager = new window.RSTH_IH.NpcManager();
        } else {
            // ★ 既存のNpcManagerがある場合はスプライトをTilemapに再付与
            window.RSTH_IH._npcManager.attachAllSprites();
        }
    };


    const _Scene_Map_terminate2 = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function () {
        if (window.RSTH_IH._npcManager) {
            window.RSTH_IH._npcManager.detachAllSprites();
        }

        _Scene_Map_terminate2.call(this);
    };

    // NPC発生チェックメイン処理
    window.RSTH_IH.isStandableTile2 = function (x, y) {
        if (!$gameMap.isValid(x, y)) return false;
        if ([1, 7].includes($gameMap.terrainTag(x, y))) return false;
        if ($gameMap.eventsXy(x, y).length > 0) return false;
        if (window.RSTH_IH._npcManager.getNpcAt(x, y)) return false;
        return [2, 4, 6, 8].some(dir => $gameMap.isPassable(x, y, dir));
    };

    // クエストでのnpc発生チェック
    window.RSTH_IH.QuestNpcSpawnCheck2 = function (count = 10) {
        // MAPのタイル数を読み込み
        const ax = $dataMap.width || 64;
        const ay = $dataMap.height || 64;

        // 初期値はマップ中央で障害物は考慮しない。
        let x = Math.round(ax / 2);
        let y = Math.round(ay / 2);
        for (let i = 0; i < count; i++) {
            x = Math.floor(Math.random() * ax);
            y = Math.floor(Math.random() * ay);

            if (!window.RSTH_IH.isStandableTile2(x, y)) continue;
            return { x, y };
        }
        return { x, y };
    };


    const _Game_Player_moveStraight = Game_Player.prototype.moveStraight;
    Game_Player.prototype.moveStraight = function (d) {
        const dx = $gameMap.roundXWithDirection(this.x, d) - this.x;
        const dy = $gameMap.roundYWithDirection(this.y, d) - this.y;
        const targetX = this.x + dx;
        const targetY = this.y + dy;

        // NPCとの衝突判定
        const npcList = Object.values(window.RSTH_IH._npcManager._npcs);
        const npc = npcList.find(n => n._x === targetX && n._y === targetY);
        if (npc && typeof npc.attemptRetreatFrom === "function") {
            const retreated = npc.attemptRetreatFrom(this.x, this.y);
            if (retreated) {
                // NPCがどいた後に再度移動を試みる
                _Game_Player_moveStraight.call(this, d);
                return;
            } else {
                // NPCがどけなかった場合は移動キャンセル
                return;
            }
        }

        // 通常移動
        _Game_Player_moveStraight.call(this, d);
    };

})();
