/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {
    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;

    // mob移動に使う方向定数dirs
    const DIRS = [
        { dx: 0, dy: -1, dir: 8 }, { dx: 1, dy: 0, dir: 6 },
        { dx: 0, dy: 1, dir: 2 }, { dx: -1, dy: 0, dir: 4 },
        { dx: 1, dy: -1, dir: 9 }, { dx: 1, dy: 1, dir: 3 },
        { dx: -1, dy: 1, dir: 1 }, { dx: -1, dy: -1, dir: 7 }
    ];

    const _Scene_Map_initialize = Scene_Map.prototype.initialize;
    Scene_Map.prototype.initialize = function () {
        _Scene_Map_initialize.call(this);
        if (!window.RSTH_IH._mobManager) {
            window.RSTH_IH._mobManager = new window.RSTH_IH.MobManager();
        }

    };


    window.RSTH_IH.spawnMob = function (definitionIndex, x, y) {
        if (!window.RSTH_IH._mobManager) {
            //console.warn("MobManager が未初期化です");
            return null;
        }
        return window.RSTH_IH._mobManager.createMob(definitionIndex, x, y);
    };


    window.RSTH_IH.MobManager = class {
        constructor() {
            this._definitions = window.RSTH_IH.MobDefinitions || [];
            this._mobs = {};
            this._mobsOrderedKeys = [];
            this._nextId = 1;
        }

        createMob(definitionIndex, x, y) {
            const definition = this._definitions[definitionIndex];
            const mobId = "MOB_" + this._nextId++;
            const mob = new window.RSTH_IH.Mob(mobId, definition, x, y);
            this._mobs[mobId] = mob;
            this._mobsOrderedKeys.push(mobId);

            while (this._mobsOrderedKeys.length > window.RSTH_IH.MaxMobAmount) {
                const oldestId = this._mobsOrderedKeys.shift();
                const mobToRemove = this._mobs[oldestId];
                if (mobToRemove?._databaseData.meta?.normb) {
                    this._mobsOrderedKeys.push(oldestId);
                    continue;
                }
                this.removeMob(oldestId);
            }

            const tilemap = SceneManager._scene?._spriteset?._tilemap;
            const sprite = mob.sprite?.();
            if (tilemap && sprite && sprite.parent !== tilemap) {
                tilemap.addChild(sprite);
            }

            if (RSTH_DEBUG_LOG) console.log("[spawnMob ]mob ", mob);
            if (RSTH_DEBUG_LOG) console.log("[spawnMob ]this._mobs", this._mobs);
            return mob;
        }

        clearAllMobs() {
            for (const mobId of this._mobsOrderedKeys) {
                this.removeMob(mobId);
            }

            this._mobs = {};
            if (RSTH_DEBUG_LOG) console.log("[MobManager] 全Mobを削除しました");
        }

        getAllMobIdsInOrder() {
            return [...this._mobsOrderedKeys];
        }

        removeMob(mobId) {
            const mob = this._mobs[mobId];
            if (mob && mob._sprite?.parent) {
                mob._sprite.parent.removeChild(mob._sprite);
            }
            delete this._mobs[mobId];
            const index = this._mobsOrderedKeys.indexOf(mobId);
            if (index !== -1) this._mobsOrderedKeys.splice(index, 1);
        }

        updateAll() {
            if (!this._mobs) return;
            const mobs = Object.values(this._mobs);
            for (const mob of mobs) {
                mob.update();
                this.checkPlayerCollision(mob);
            }

            for (const mob of mobs) {
                const sprite = mob._sprite;
                if ((sprite?._isDying || mob._isDead) && !sprite?.parent) {
                    this.removeMob(mob._id);
                }
            }
        }

        checkPlayerCollision(mob) {
            const px = $gamePlayer.x;
            const py = $gamePlayer.y;
            const dx = Math.abs(mob.x() - px);
            const dy = Math.abs(mob.y() - py);
            if (dx <= 1 && dy <= 1) {
                //console.log(`プレイヤーとMob(${mob._id})が接近しています`);
            }
        }

        hasMobAt(x, y, excludeId = null) {
            return Object.values(this._mobs).some(mob => {
                if (excludeId && mob._id === excludeId) return false;
                return mob._x === x && mob._y === y;
            });
        }

        serialize() {
            return Object.values(this._mobs).map(mob => mob.serialize());
        }

        deserialize(dataList) {
            for (const data of dataList) {
                const definition = this._definitions[data.definitionIndex];
                if (!definition) continue;
                const mob = new window.RSTH_IH.Mob(data.id, definition, data.x, data.y);
                mob.deserialize(data);
                this._mobs[data.id] = mob;
                this._mobsOrderedKeys.push(data.id);
            }
        }

        attachAllSprites() {
            const tilemap = SceneManager._scene?._spriteset?._tilemap;
            if (!tilemap) return;
            for (const mob of Object.values(this._mobs)) {
                const sprite = mob.sprite?.();
                if (sprite && sprite.parent !== tilemap) {
                    tilemap.addChild(sprite);
                }
            }
        }

        detachAllSprites() {
            const tilemap = SceneManager._scene?._spriteset?._tilemap;
            if (!tilemap) return;
            for (const mob of Object.values(this._mobs)) {
                const sprite = mob.sprite?.();
                if (sprite && sprite.parent === tilemap) {
                    tilemap.removeChild(sprite);
                }
            }
        }

        canMoveToSimple(tx, ty, excludeMobId = null, dir = 2) {
            if (tx < 0 || ty < 0 || tx >= $gameMap.width() || ty >= $gameMap.height()) return false;
            if (!$gameMap.isPassable(tx, ty, dir)) return false;
            if ($gameMap.eventsXy(tx, ty).length > 0) return false;
            if ($gamePlayer.x === tx && $gamePlayer.y === ty) return false;
            if (this.hasMobAt(tx, ty, excludeMobId)) return false;
            return true;
        }

        getMobAt(x, y) {
            return Object.values(this._mobs).find(mob => mob._x === x && mob._y === y) || null;
        }
    };

    window.RSTH_IH.Mob = class {
        constructor(mobId, definition, x, y) {
            this._type = "mob";
            this._id = mobId;
            this._definition = definition;
            this._x = x;
            this._y = y;
            this._lv = definition.defaultLv;
            this._exp = definition.defaultExp;
            this._isEnemy = definition.isEnemy;
            this._hitStop = 0;
            this._attackCooldown = 0;
            this._castTime = 0;
            this._castSkill = null;
            this._stateEffectTimes = 0;
            this.pendingHits = [];
            this.pendingHitTimer = 0; // フレームカウント
            this._target = null;
            this._healed = false;

            this._aiMode = "random";  // 追加: AI行動モード（"random" or "chase"）
            //console.log("this._definition", this._definition);

            const db = this._isEnemy ? $dataEnemies[definition.databaseId] : $dataActors[definition.databaseId];
            //console.log("db", db);
            this._databaseData = db;

            this._traits = db.traits || [];
            this._skills = db.skills || [];

            this._sprite = new window.RSTH_IH.Sprite_Mob(this);
            this._moveCooldown = 1;
            this._direction = 2;
            this._pattern = 1;
            this._animationCount = 0;
            this._realX = x;
            this._realY = y;
            this._moveSpeed = definition.moveSpd;
            this._isDead = false;
            this._states = [];  // ステートIDの配列
            this._boss = false;
            this._nokb = false;
            if (db.meta) {
                if (db.meta.boss) {
                    this._boss = db.meta.boss;
                }
                if (db.meta.nokb) {
                    this._nokb = true;
                }
            }
            //console.log("db.meta", db.meta);
            //console.log("this._boss", this._boss);

            if (db.params) {
                this._maxhp = db.params[0];
                this._hp = db.params[0];
                this._maxmp = db.params[1];
                this._mp = db.params[1];
                this._atk = db.params[2];
                this._def = db.params[3];
                this._matk = db.params[4];
                this._dex = db.params[5];
                this._agi = db.params[6];
                this._luk = db.params[7];
            }
            this._rewardExp = db.exp;
            this._rewardGold = db.gold;

            // Mobクラスのconstructor内（定義取得のあとなど）
            this._weaponChances = [];
            this._weaponChancesOrg = [];
            this._attackDistance = 1; // デフォルト攻撃距離

            const metaDistance = db.note?.match(/<distance:(\d+)>/);
            if (metaDistance) {
                this._attackDistance = Number(metaDistance[1]);
            }

            const note = db.note || "";
            const matches = note.matchAll(/<weaponId:(\d+),(\d+)>/g);
            for (const m of matches) {
                const weaponId = Number(m[1]);
                const chance = Number(m[2]);
                if (!isNaN(weaponId) && !isNaN(chance)) {
                    this._weaponChances.push({ weaponId, chance });
                    this._weaponChancesOrg.push({ weaponId, chance });
                }
            }

            this._castBar = new window.RSTH_IH.Sprite_CastBar(60, 6);
            this._castBar.y = -80; // skill名ポップアップより下に
            this._castBar.visible = false;
            this._sprite.addChild(this._castBar);


        }

        queueHit(weapon, knockback, sourceX, sourceY, source) {
            // ノックバックは即時
            const mobManager = window.RSTH_IH._mobManager;

            if (!this._nokb) {
                window.RSTH_IH.applyKnockback(this, sourceX, sourceY, knockback, mobManager);
            }

            //console.log("[mob.queueHit]source", source);

            let dmg = 0;
            // ダメージだけ先に計算して貯める
            if (!source) {
                dmg = window.RSTH_IH.damage_calculation(weapon, this, null, "actor", "mob");
            }
            else if (source._type === "npc") {
                dmg = window.RSTH_IH.damage_calculation(weapon, this, source, "npc", "mob");
            }
            this.pendingHits.push(dmg);

            this.pendingHitTimer = 24; // 0.2秒
        }

        applyPendingHits() {
            if (this.pendingHits.length === 0) return;

            // ダメージ合計
            const totalDamage = this.pendingHits.reduce((sum, v) => sum + v, 0);

            // Pop 表示だけここでまとめて出す
            this.showDamagePopup(totalDamage);

            // リセット
            this.pendingHits = [];

        }

        update() {
            this.updateStateEffectTimers();
            if (this._isDead) {
                if (this._sprite) this._sprite.update();
                return;  // 死亡中はAI・移動処理をスキップ
            }
            if (this.hasState(12)) {// 麻痺
                if (this._sprite) this._sprite.update();  // スプライトは更新し続ける
                return;
            }
            if (this._hitStop > 0) {
                this._hitStop--;
                if (this._sprite) this._sprite.update();  // スプライトは更新し続ける
                return;
            }
            if (this._attackCooldown > 0) {
                this._attackCooldown--;
            }
            if (this._castTime > 0) {
                this._castTime--;

                if (this._castBar) {
                    this._castBar.visible = true;
                    this._castBar.updateProgress(this._castTime);
                }

                if (this._castTime === 0 && this._castSkill) {
                    window.RSTH_IH.showSkillNamePopup(this, this._castSkill);
                    // function (mob, npc, skill, from = "player")
                    window.RSTH_IH.executeSkill(this._target, this, this._castSkill, "mob");
                    this._castSkill = null;
                    if (this._castBar) this._castBar.visible = false;
                }

                if (this._sprite) this._sprite.update();
                return;
            }

            if (this.pendingHitTimer > 0) {
                this.pendingHitTimer--;
                if (this.pendingHitTimer <= 0) {
                    this.applyPendingHits();
                }
            }

            this.updateAI();
            this.updateMove();
            this.updateAnimation();
            if (this._sprite) this._sprite.update();
        }

        updateAI() {
            if (this.isMoving()) return;
            if (this._moveCooldown > 0) { this._moveCooldown--; return; }

            const offset = this.distancePerFrame(); // ③ キャッシュ
            const playerX = $gamePlayer.x;
            const playerY = $gamePlayer.y;

            let dx = null;
            let dy = null;
            let npcflag = false;
            let npcx = null;
            let npcy = null;

            if (window.RSTH_IH._npcManager._npcs && Object.keys(window.RSTH_IH._npcManager._npcs).length > 0) {
                const firstNpcKey = Object.keys(window.RSTH_IH._npcManager._npcs)[0];
                const firstNpc = window.RSTH_IH._npcManager._npcs[firstNpcKey];
                this._target = firstNpc;
                npcx = firstNpc._x;
                npcy = firstNpc._y;
                dx = Math.abs(npcx - this._x);
                dy = Math.abs(npcy - this._y);
                npcflag = true;
            } else {
                this._target = $gamePlayer;
                dx = Math.abs(playerX - this._x);
                dy = Math.abs(playerY - this._y);
            }

            if (this._aiMode === "random") {
                const dir = DIRS[Math.floor(Math.random() * DIRS.length)];

                const prevDir = this._direction; // ② 変化検出
                if (dir.dx !== 0 && dir.dy !== 0) {
                    if (Math.abs(dir.dx) >= Math.abs(dir.dy)) {
                        this._direction = (dir.dx > 0) ? 6 : 4;
                    } else {
                        this._direction = (dir.dy > 0) ? 2 : 8;
                    }
                } else {
                    if (dir.dx !== 0) {
                        this._direction = (dir.dx > 0) ? 6 : 4;
                    } else if (dir.dy !== 0) {
                        this._direction = (dir.dy > 0) ? 2 : 8;
                    }
                }
                if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新

                const tx = this._x + dir.dx;
                const ty = this._y + dir.dy;

                if (this.canMoveTo(tx, ty, dir.dir)) {
                    this._x = tx;
                    this._y = ty;
                    if (this._realX === this._x - dir.dx) this._realX += dir.dx * offset;
                    if (this._realY === this._y - dir.dy) this._realY += dir.dy * offset;
                }
            }

            else if (this._aiMode === "chase") {
                const attackRange = this._attackDistance ?? 1;
                const targetsInRange = [];

                // 1. NPCの範囲内検索
                for (const key of Object.keys(window.RSTH_IH._npcManager._npcs)) {
                    const npc = window.RSTH_IH._npcManager._npcs[key];
                    const dist = Math.abs(npc._x - this._x) + Math.abs(npc._y - this._y);
                    if (dist > 0 && dist <= attackRange) {
                        targetsInRange.push({ target: npc, dist, type: "npc" });
                    }
                }

                // 2. プレイヤーも範囲内かチェック
                const playerDist = Math.abs(playerX - this._x) + Math.abs(playerY - this._y);
                if (playerDist > 0 && playerDist <= attackRange) {
                    targetsInRange.push({ target: $gamePlayer, dist: playerDist, type: "player" });
                }

                // 3. 攻撃対象決定（優先度：距離が近いNPC → プレイヤー）
                if (targetsInRange.length > 0) {
                    targetsInRange.sort((a, b) => a.dist - b.dist || (a.type === "npc" ? -1 : 1));
                    const chosen = targetsInRange[0];

                    //console.log("chosen", chosen);
                    //console.log("targetsInRange", targetsInRange);

                    if (this._attackCooldown <= 0) {
                        if (chosen.type === "npc") {
                            this.attackNpc(chosen.target);
                        } else {
                            this.attackPlayer(); // 既存の攻撃処理
                        }
                        this._attackCooldown = this.calculateAtkSpd();
                    }
                    return;
                }

                // 攻撃範囲に誰もいない：プレイヤーを追跡
                const randIndices = [0, 1, 2, 3, 4, 5, 6, 7].sort(() => Math.random() - 0.5);
                let bestDir = null;
                let bestDist = Infinity;

                for (let i = 0; i < randIndices.length; i++) {
                    const dir = DIRS[randIndices[i]];
                    const tx = this._x + dir.dx;
                    const ty = this._y + dir.dy;
                    if (!this.canMoveTo(tx, ty, dir.dir)) continue;

                    const d = Math.abs(playerX - tx) + Math.abs(playerY - ty);
                    if (d < bestDist) {
                        bestDist = d;
                        bestDir = dir;
                    }
                }

                if (bestDir) {
                    const prevDir = this._direction;
                    if (bestDir.dx !== 0 && bestDir.dy !== 0) {
                        if (Math.abs(bestDir.dx) >= Math.abs(bestDir.dy)) {
                            this._direction = (bestDir.dx > 0) ? 6 : 4;
                        } else {
                            this._direction = (bestDir.dy > 0) ? 2 : 8;
                        }
                    } else {
                        if (bestDir.dx !== 0) {
                            this._direction = (bestDir.dx > 0) ? 6 : 4;
                        } else if (bestDir.dy !== 0) {
                            this._direction = (bestDir.dy > 0) ? 2 : 8;
                        }
                    }
                    if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction);
                    this._x += bestDir.dx;
                    this._y += bestDir.dy;
                    this._realX += bestDir.dx * offset;
                    this._realY += bestDir.dy * offset;
                }

                this._moveCooldown = 1;
            }


            else if (this._aiMode === "shooter") {
                const attackRange = this._attackDistance ?? 3;

                const targetsInRange = [];

                if (this._databaseData.id === 63 && this._healed === false) {

                }


                // 1. NPCの範囲内検索
                for (const key of Object.keys(window.RSTH_IH._npcManager._npcs)) {
                    const npc = window.RSTH_IH._npcManager._npcs[key];
                    const dist = Math.abs(npc._x - this._x) + Math.abs(npc._y - this._y);
                    if (dist > 0 && dist <= attackRange) {
                        targetsInRange.push({ target: npc, dist, type: "npc" });
                    }
                }

                // 2. プレイヤーも範囲内かチェック
                const playerDist = Math.abs(playerX - this._x) + Math.abs(playerY - this._y);
                if (playerDist > 0 && playerDist <= attackRange) {
                    targetsInRange.push({ target: $gamePlayer, dist: playerDist, type: "player" });
                }

                // 3. 攻撃対象決定（優先度：距離が近いNPC → プレイヤー）
                if (targetsInRange.length > 0) {
                    targetsInRange.sort((a, b) => a.dist - b.dist || (a.type === "npc" ? -1 : 1));
                    const chosen = targetsInRange[0];
                    if (this._attackCooldown <= 0) {
                        if (chosen.type === "npc") {
                            this._target = chosen.target;
                            if (this._databaseData.id === 63) {
                                this.bossAttack(this._databaseData.id);
                            } else {
                                //console.log("chosen.target", chosen.target);
                                this.attackNpc(chosen.target);
                            }
                        } else {
                            if (this._databaseData.id === 63) {
                                this.bossAttack(this._databaseData.id);
                            } else {
                                //console.log("chosen.target", chosen.target);
                                //console.log("[attackPlayer]");
                                this._target = chosen.target;
                                this.attackPlayer();
                            }
                        }
                        this._attackCooldown = this.calculateAtkSpd();
                    }
                    return;
                }

                /*
                if (dist <= attackRange && dist > 0) {
                    if (this._attackCooldown <= 0) {
                        if (this._databaseData.id === 63) {
                            this.bossAttack(this._databaseData.id);
                        } else {
                            if (npcflag) {
                                this.attackNpc();
                            } else {
                                this.attackPlayer();
                            }
                        }
                        this._attackCooldown = this.calculateAtkSpd();
                    }
                }
                */

                const prevDir = this._direction;
                if (dx > dy) {
                    this._direction = (playerX > this._x) ? 6 : 4;
                } else {
                    this._direction = (playerY > this._y) ? 2 : 8;
                }
                if (this._direction !== prevDir) this._sprite.updatePattern(this._pattern, this._direction); // ② 条件付き更新
            }

            this._moveCooldown = 1;
        }

        updateMove() {
            let distancePerFrame = 0;
            if (this.hasState(10)) {// 鈍足
                distancePerFrame = 1 / 256;
            } else {
                distancePerFrame = this._moveSpeed / 256;

            }
            const dx = this._x - this._realX;
            const dy = this._y - this._realY;
            if (Math.abs(dx) > 0) this._realX += Math.sign(dx) * Math.min(Math.abs(dx), distancePerFrame);
            if (Math.abs(dy) > 0) this._realY += Math.sign(dy) * Math.min(Math.abs(dy), distancePerFrame);
        }


        isMoving() {
            return (Math.abs(this._realX - this._x) > 0.001 || Math.abs(this._realY - this._y) > 0.001);
        }

        updateAnimation() {
            if (this.isMoving()) {
                this._animationCount++;
                if (this._animationCount >= 15) {
                    this._pattern = (this._pattern % 3) + 1;
                    this._sprite.updatePattern(this._pattern, this._direction);
                    this._animationCount = 0;
                }
            } else {
                this._pattern = 1;
                this._sprite.updatePattern(this._pattern, this._direction);
            }
        }

        canMoveTo(tx, ty, dir) {
            if (tx < 0 || ty < 0 || tx >= $gameMap.width() || ty >= $gameMap.height()) return false;
            if (!$gameMap.isPassable(tx, ty, dir)) return false;
            const events = $gameMap.eventsXy(tx, ty);
            if (events.length > 0) return false;
            if ($gamePlayer.x === tx && $gamePlayer.y === ty) return false;
            if (window.RSTH_IH._mobManager.hasMobAt(tx, ty, this._id)) return false;
            return true;
        }

        attackPlayer() {
            this._direction = this.calculateDirectionToPlayer();
            this._sprite.updatePattern(this._pattern, this._direction);

            const actions = this._databaseData.actions || [];
            let performedAction = false;

            // 1. まず通常攻撃（skillId:1）を評価
            const normalAttack = actions.find(a => a.skillId === 1);
            if (normalAttack) {
                const rating = normalAttack.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    // 通常攻撃成功
                    this._sprite.startRedFlash();
                    for (const entry of this._weaponChances) {
                        if (Math.random() * 100 < entry.chance) {
                            const weapon = $dataWeapons[entry.weaponId];
                            //console.log("[attackPlayer]this._target", this._target);
                            performedAction = this.useWeapons(weapon);
                        }
                    }

                    if (!performedAction) {
                        // 通常攻撃失敗（確率外）→素手攻撃
                        //console.log("[attackPlayer !performedAction]this._target", this._target);
                        window.RSTH_IH.damage_calculation(null, this, null, "mob", "actor");
                    }

                    return;
                }
            }

            // 2. 通常攻撃に失敗したら、他のスキルを順に評価
            for (const action of actions) {
                if (action.skillId === 1) continue; // 通常攻撃は除外

                const rating = action.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    const skill = $dataSkills[action.skillId];
                    //(`[MobSkill] skill`, skill);
                    const castTag = skill.meta?.casttime;
                    const castFrames = castTag ? Number(castTag) : 0;

                    if (castFrames > 0) {
                        this._castTime = castFrames;
                        this._castSkill = skill;


                        if (this._castBar) {
                            this._castBar.setup(castFrames);
                            this._castBar.visible = true;
                        }
                    } else {
                        this._sprite.startRedFlash();
                        //console.log(`[MobSkill] action`, action);
                        //console.log(`[MobSkill] Using Skill ID: ${action.skillId}, rating: ${rating}, rand: ${rand}`);
                        window.RSTH_IH.showSkillNamePopup(this, skill);
                        // function (target, source, skill, from = "player")
                        window.RSTH_IH.executeSkill(this._target, this, skill, "mob");
                    }

                    performedAction = true;
                    break;
                }
            }

            // 3. どれも成功しなかった場合は何もしない
            if (!performedAction) {
                //console.log("[MobSkill] どのスキルも評価失敗 → 行動しない");
            }
        }

        attackNpc(npc) {
            this._direction = this.calculateDirectionToPlayer();
            this._sprite.updatePattern(this._pattern, this._direction);

            const actions = this._databaseData.actions || [];
            let performedAction = false;

            //console.log("this._target", this._target);
            //console.log("npc", npc);
            // 1. まず通常攻撃（skillId:1）を評価
            const normalAttack = actions.find(a => a.skillId === 1);
            if (normalAttack) {
                const rating = normalAttack.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    // 通常攻撃成功
                    this._sprite.startRedFlash();
                    for (const entry of this._weaponChances) {
                        if (Math.random() * 100 < entry.chance) {
                            const weapon = $dataWeapons[entry.weaponId];
                            //console.log("[attackNpc]this._target", this._target);
                            performedAction = this.useWeapons(weapon);
                            //console.log("performedAction", performedAction);
                            //console.log("weapon", weapon);
                        }
                    }

                    if (!performedAction) {
                        // 通常攻撃失敗（確率外）→素手攻撃
                        // function (item, mob, npc, from = "actor", target = "mob", bowatk = 0)
                        const dmg = window.RSTH_IH.damage_calculation(null, this, npc, "mob", "npc", 0);
                        //console.log("dmg", dmg);
                        //console.log("npc", npc);
                        npc.showDamagePopup(dmg);
                    }

                    return;
                }
            }

            // 2. 通常攻撃に失敗したら、他のスキルを順に評価
            for (const action of actions) {
                if (action.skillId === 1) continue; // 通常攻撃は除外

                const rating = action.rating;
                const rand = Math.random() * 10;
                if (rand < rating) {
                    const skill = $dataSkills[action.skillId];
                    //(`[MobSkill] skill`, skill);
                    const castTag = skill.meta?.casttime;
                    const castFrames = castTag ? Number(castTag) : 0;

                    if (castFrames > 0) {
                        this._castTime = castFrames;
                        this._castSkill = skill;


                        if (this._castBar) {
                            this._castBar.setup(castFrames);
                            this._castBar.visible = true;
                        }
                    } else {
                        this._sprite.startRedFlash();
                        //console.log(`[MobSkill] action`, action);
                        //console.log(`[MobSkill] Using Skill ID: ${action.skillId}, rating: ${rating}, rand: ${rand}`);
                        window.RSTH_IH.showSkillNamePopup(this, skill);
                        // function (target, source, skill, from = "player")
                        window.RSTH_IH.executeSkill(this._target, this, skill, "mob");
                    }

                    performedAction = true;
                    break;
                }
            }

            // 3. どれも成功しなかった場合は何もしない
            if (!performedAction) {
                //console.log("[MobSkill] どのスキルも評価失敗 → 行動しない");
            }
        }

        bossAttack(boss_id) {
            this._direction = this.calculateDirectionToPlayer();
            this._sprite.updatePattern(this._pattern, this._direction);

            let weapons = this._weaponChances;

            // weapons が空の場合は武器リストを初期化
            if (weapons.length === 0) {

                // 武器リスト初期化
                this._weaponChances = JSON.parse(JSON.stringify(this._weaponChancesOrg));
                weapons = this._weaponChances;
            }

            // weapons が空でないなら先頭要素を取り出して usedWeapons に追加
            if (weapons.length > 0) {
                this._sprite.startRedFlash();
                const weapon = $dataWeapons[weapons[0].weaponId];
                this.useWeapons(weapon);

                weapons.shift(); // 先頭要素を取り出して削除
            }
        }

        useWeapons(weapon) {
            if (weapon) {
                if (!this._target) return false;
                const isProjectile = !!weapon.meta?.projectile;
                const startX = this._x;
                const startY = this._y;
                let targetX = null;
                let targetY = null;
                //console.log("this._target", this._target);
                if (this._target?._actorId) {
                    targetX = $gamePlayer.x;
                    targetY = $gamePlayer.y;
                } else {
                    targetX = this._target._x;
                    targetY = this._target._y;
                }
                // if (!targetX || !targetY) return false;

                const meta = weapon.meta;
                const atk = this._atk;
                const power = weapon.params[2] || 5;

                if (isProjectile) {
                    if (meta.boomerang) {
                        const maxDistance = Number(meta.maxdistance) || 8;
                        window.RSTH_IH.ProjectileManager.createBoomerang(
                            weapon, startX, startY, targetX, targetY,
                            atk, maxDistance, "mob", this
                        );
                    }
                    else if (meta.knife) {
                        const knifeIds = new Set([43, 67, 68, 69, 72]);
                        const bowIds = new Set([38, 39, 40, 41, 42]);

                        // 投げナイフ
                        if (knifeIds.has(weapon.id)) {
                            const knifeAmounts = { 67: 2, 68: 3, 69: 4, 72: 5 };
                            const amount = knifeAmounts[weapon.id] || 1;
                            for (let i = 0; i < amount; i++) {
                                window.RSTH_IH.ProjectileManager.createKnife(
                                    weapon,
                                    startX, startY,
                                    targetX, targetY,
                                    atk,
                                    15, "mob", this
                                );
                            }
                        }
                        // 天弓
                        else if (bowIds.has(weapon.id)) {
                            const bowAmounts = { 39: 2, 40: 3, 41: 4, 42: 5 };
                            const amount = bowAmounts[weapon.id] || 1;
                            for (let i = 0; i < amount; i++) {
                                window.RSTH_IH.ProjectileManager.createKnife(
                                    weapon,
                                    startX, startY,
                                    targetX, targetY,
                                    atk,
                                    20, "mob", this
                                );
                            }

                        }
                    }
                    // split
                    else if (meta.split) {
                        const splitIds = new Set([73, 74, 75, 76, 77]);
                        if (splitIds.has(weapon.id)) {
                            window.RSTH_IH.ProjectileManager.createSplit(
                                weapon,
                                startX, startY,
                                targetX, targetY,
                                atk,
                                20, "mob", this
                            );
                        }
                    }
                    else if (meta.banana) {
                        const range = 20;
                        window.RSTH_IH.ProjectileManager.createBananaSpread(weapon, power, range, "mob", this);
                    }
                    else {
                        let item = $dataItems[this._databaseData.meta?.arrowId || 40];

                        window.RSTH_IH.ProjectileManager.createArrow(
                            startX, startY, targetX, targetY, item, power, "mob", this
                        );
                    }
                } else {
                    window.RSTH_IH.useWeaponByMob(this, weapon);
                }
                return true;
            }
            return false;
        }


        addState(stateId) {
            const state = $dataStates[stateId];
            if (!this._states.includes(stateId)) {
                //console.log("stateId", stateId);
                this._states.push(stateId);
                //console.log(`[Mob] ステート ${stateId} を付与`);
                // 必要に応じてステート効果を即時適用
                // 例: パラメータ上昇・ビジュアル変更・状態表示など
            }
            if (!this._stateEffectTimes) this._stateEffectTimes = {};
            let effectTime = 0;
            if (state.meta.et === "ex") {
                effectTime = 9999999;
            } else {
                effectTime = Number(state.meta.et || 0);
            }
            if (effectTime > 0) {
                this._stateEffectTimes[stateId] = effectTime;
                //console.log(`[addState] ステート${state.name} に制限時間 ${effectTime} フレーム設定`);
            }
        }

        removeState(stateId) {
            const index = this._states.indexOf(stateId);
            if (index >= 0) {
                this._states.splice(index, 1);
                //console.log(`[Mob] ステート ${stateId} を解除`);
                // 効果の解除処理もここに追加可能
            }
        }

        hasState(stateId) {
            return this._states.includes(stateId);
        }

        updateStateEffectTimers() {
            if ($gameMessage.isBusy()) return;

            const times = this._stateEffectTimes || {};
            const removed = [];

            for (const stateId of this._states) {
                if (!(stateId in times)) continue;

                times[stateId]--;
                if (times[stateId] <= 0) {
                    removed.push(stateId);
                }
            }

            removed.forEach(stateId => {
                this.removeState(stateId);
                delete times[stateId];
                const state = $dataStates[stateId];
                //console.log(`[updateStateEffectTimers] ${state.name} の効果が切れた`);
            });
        }

        calculateDirectionToPlayer() {
            const dx = $gamePlayer.x - this._x;
            const dy = $gamePlayer.y - this._y;

            if (dx === 0 && dy === 0) {
                return 2; // 同じ位置なら下（保険）
            }

            if (dy > 0) {
                // プレイヤーがmobより下側にいる場合
                return 2;
            }

            if (dy < 0) {
                // プレイヤーがmobより上側にいる場合
                return 8; // 真上にいる
            }

            // dy==0 で左右だけにいるとき
            if (dx > 0) return 6;
            if (dx < 0) return 4;

            return 2; // 万が一
        }

        calculateDirectionToTarget(target) {
            // mob npc専用
            //console.log("target", target);
            const dx = target._x - this._x;
            const dy = target._y - this._y;

            if (dx === 0 && dy === 0) {
                return 2; // 同じ位置なら下（保険）
            }

            if (dy > 0) {
                return 2;
            }

            if (dy < 0) {
                return 8; // 真上にいる
            }

            // dy==0 で左右だけにいるとき
            if (dx > 0) return 6;
            if (dx < 0) return 4;

            return 2; // 万が一
        }

        calculateAtkSpd() {
            const minct = 10;
            const aspd = window.RSTH_IH.calculateAtkSpd(this._agi);
            const ct = 120;
            let atkSpd = Math.round(ct - ct * (aspd / 100));
            if (atkSpd < minct) atkSpd = minct;
            return atkSpd;
        }


        distancePerFrame() { return this._moveSpeed / 256; }
        sprite() { return this._sprite; }
        x() { return this._x; }
        y() { return this._y; }

        serialize() {
            return {
                id: this._id, definitionIndex: window.RSTH_IH.MobDefinitions.indexOf(this._definition),
                x: this._x, y: this._y, lv: this._lv, exp: this._exp
            };
        }

        deserialize(data) { this._lv = data.lv; this._exp = data.exp; }

        takeDamage(amount, cri = 0) {
            if (!this._isDead) {
                this._hp -= amount;
                this._sprite?.startShake();
                if (!this._boss) {
                    if (cri === 1) {
                        this._hitStop = 15;
                    } else {
                        this._hitStop = 5;
                    }
                }
                if (this._hp <= 0) {
                    this._hp = 0;
                    this.die();
                }

            }
        }

        showDamagePopup(amount, cri = 0) {
            if (!this._sprite) return;
            if (!window.RSTH_IH.DmgPopup) return;

            const seId = (cri === 0) ? 10 : 14;
            AudioManager.playSe($dataSystem.sounds[seId]);
            const popup = new window.RSTH_IH.Sprite_DamagePopup(amount, cri);
            this._sprite.addChild(popup);
        }

        die() {
            this._isDead = true;
            this._sprite?.startDeathAnimation();
            window.RSTH_IH.KilledMobCount += 1;

            if (window.RSTH_IH.KilledMobCount > 0 && !ConfigManager["achi_FKL"]) {
                window.RSTH_IH.Window_CheckAchievement(0, 0, 1);
            }
            const thresholds = [1000, 3000, 5000];
            for (const t of thresholds) {
                if (window.RSTH_IH.KilledMobCount >= t && !ConfigManager[`achi_MKL${t}`]) {
                    window.RSTH_IH.Window_CheckAchievement(0, 0, t);
                }
            }

            if (window.RSTH_IH.KilledMobCount > 0 && window.RSTH_IH.KilledMobCount % 1000 === 0) {
                const unp = 10;
                window.RSTH_IH.unshoP += unp;
                ConfigManager.unshoP = window.RSTH_IH.unshoP;
                ConfigManager.save();

                const text = `<<敵を1000体討伐した為うんしょポイントを${unp}獲得した！>>`;
                window.RSTH_IH.LogWindowManager.addLog(text, 2);
            }

            if (this._boss) {
                let text = `<<BOSS級の敵を倒した！>>`;
                window.RSTH_IH.LogWindowManager.addLog(text, 1);

                // うんしょポイント獲得
                const unp = 25;
                window.RSTH_IH.unshoP += unp;
                ConfigManager.unshoP = window.RSTH_IH.unshoP;
                ConfigManager.save();

                text = `<<うんしょポイントを${unp}獲得した！>>`;
                window.RSTH_IH.LogWindowManager.addLog(text, 2);

                //console.log("this._boss", this._boss);
                //アチーブメントチェック
                window.RSTH_IH.Window_CheckAchievement(this._boss);

            }
            const actor = $gameParty.leader();
            if (actor) {
                actor.gainExp(this._rewardExp);
                if (RSTH_DEBUG_LOG) console.log(`[DropManager] 経験値 ${exp} 獲得（アイテム即削除）`);
            }

            AudioManager.playSe($dataSystem.sounds[11]);

            if (!$gameSystem._treasureQueueCount) $gameSystem._treasureQueueCount = 0;
            if (!$gameSystem._defeatedTreasureMobCount) $gameSystem._defeatedTreasureMobCount = 0;

            const tmc = window.RSTH_IH.TreasureMobCount;
            $gameSystem._defeatedTreasureMobCount++;
            if ($gameSystem._defeatedTreasureMobCount >= tmc) {
                const times = Math.floor($gameSystem._defeatedTreasureMobCount / tmc);
                $gameSystem._treasureQueueCount += times;
                $gameSystem._defeatedTreasureMobCount -= times * tmc;
            }


            // ドロップ処理
            const dropItems = this._databaseData.dropItems;
            if (Array.isArray(dropItems) && dropItems.length > 0) {
                for (const i of dropItems) {
                    const dropId = i.dataId;
                    let item = null;
                    if (i.kind === 1) item = $dataItems[dropId];
                    else if (i.kind === 2) item = $dataWeapons[dropId];
                    else if (i.kind === 3) item = $dataArmors[dropId];
                    else continue;

                    if (item && Math.random() < (1 / i.denominator)) {
                        window.RSTH_IH.DropManager.dropItemSmart(this._x, this._y, item);
                    }
                }
            }
        }
    };

    Scene_Map.prototype.getTreasureUpdate = function () {
        if ($gameSystem._treasureQueueCount > 0 && !SceneManager.isSceneChanging()) {
            $gameSystem._treasureQueueCount--;

            const se = {
                name: "Open3",
                volume: 25,
                pitch: 100,
                pan: 0
            };
            AudioManager.playSe(se);
            window.RSTH_IH.showScreenBorder(1);

            if (ConfigManager.treasureEffectMode === "esy") {
                window.RSTH_IH.TreasureEasy();
            } else {
                SceneManager.push(window.RSTH_IH.Scene_Treasure);
            }
        }

    }

    window.RSTH_IH.Sprite_Mob = class extends Sprite {
        constructor(mob) {
            super();
            this._mob = mob;
            this._loaded = false;
            this._shakeX = 0;
            this._shakeY = 0;
            this._shakePower = 0;

            const definition = mob._definition;
            const bitmap = ImageManager.loadCharacter(definition.characterName);
            this.bitmap = bitmap;

            bitmap.addLoadListener(() => {
                if (window.RSTH_IH.MobD_mzonoff) {
                    // MZ形式: 8キャラシート
                    this._isBig = (definition.characterName || "").startsWith('$');
                    this._index = definition.characterIndex;
                    this._pw = this._isBig ? bitmap.width / 3 : bitmap.width / 12;
                    this._ph = this._isBig ? bitmap.height / 4 : bitmap.height / 8;
                    this._sxBase = (this._index % (this._isBig ? 1 : 4)) * 3 * this._pw;
                    this._syBase = Math.floor(this._index / (this._isBig ? 1 : 4)) * 4 * this._ph;
                } else {
                    // 1キャラシート: 常に大キャラ扱い
                    this._isBig = true;
                    this._index = 0;
                    this._pw = bitmap.width / 3;   // 3フレーム
                    this._ph = bitmap.height / 4;  // 4方向
                    this._sxBase = 0;
                    this._syBase = 0;
                }

                this.z = 4;
                this._loaded = true;
                this.updatePattern(1, 2);
            });

            this.anchor.x = 0.3;
            this.anchor.y = 1.1;

            // HPバーはbossのみ
            //console.log("mob", mob);
            this._boss = mob._databaseData?.meta?.boss;
            if (this._boss) {
                this._hpBarSprite = new Sprite(new Bitmap(60, 8));
                this.addChild(this._hpBarSprite);
                this._lastHp = -1;
            }

        }

        startShake() { this._shakePower = 25; }
        startRedFlash() { this._redFlashDuration = 10; }

        update() {
            if (!this._loaded) return;
            //if (!this._loaded || !this.bitmap || !this.bitmap.isReady()) return;
            super.update();

            if (this._boss) {
                this.updateHpBar();
            }

            if (this._isDying) {
                this.updateDeathAnimation();
            } else {
                this.updateShake();
                this.updatePosition();
                // this.updateBreathScale();

                if (this._redFlashDuration > 0) {
                    this._redFlashDuration--;
                    this.setBlendColor([255, 64, 64, 160]); // 赤フラッシュ
                } else if (this._mob.hasState && this._mob.hasState(12)) {
                    // ✅ ステート12のとき青色に
                    this.setBlendColor([64, 64, 255, 160]);
                } else if (this._mob.hasState && this._mob.hasState(10)) {
                    // ✅ ステート10のとき紫色に
                    this.setBlendColor([150, 32, 150, 160]);
                } else {
                    this.setBlendColor([0, 0, 0, 0]); // 通常
                }
            }
        }

        updateHpBar() {
            const mob = this._mob;
            if (!mob) return;

            if (this._lastHp === mob._hp) return; // HP変化なしなら更新不要

            this._lastHp = mob._hp;

            const bitmap = this._hpBarSprite.bitmap;
            bitmap.clear();

            const maxWidth = 60;
            const hpRate = mob._hp / mob._maxhp;

            // 背景
            bitmap.fillRect(0, 0, maxWidth, 8, "#000000");

            // HPバー（色はプレイヤー用と同じ）
            bitmap.gradientFillRect(
                0, 0,
                Math.floor(maxWidth * hpRate),
                8,
                ColorManager.hpGaugeColor1(),
                ColorManager.hpGaugeColor2()
            );

            // HPバーの位置（キャラクターの上）
            this._hpBarSprite.x = -6; // 幅の半分で中央合わせ
            this._hpBarSprite.y = -8;  // キャラクターの真下に12px程度オフセット

        }

        updateBreathScale() {
            // 周期的に scale を変動させる（歩行時の揺れを想定）
            const t = Graphics.frameCount / 10;  // 周期を決める
            const scaleFactor = 1 + Math.sin(t) * 0.02; // ±2%程度
            this.scale.x = scaleFactor;
            this.scale.y = scaleFactor;
        }


        updateShake() {
            if (this._shakePower > 0) {
                this._shakeX = (Math.random() - 0.5) * this._shakePower;
                this._shakeY = (Math.random() - 0.5) * this._shakePower;
                this._shakePower -= 1;
            } else {
                this._shakeX = 0; this._shakeY = 0;
            }
        }

        updatePosition() {
            const tw = $gameMap.tileWidth();
            const th = $gameMap.tileHeight();
            const dx = $gameMap.displayX();
            const dy = $gameMap.displayY();

            const baseY = this._mob._realY;

            const bobbing = Math.sin(Graphics.frameCount / 10) * 3; // ↑±2px上下に

            this.x = Math.round((this._mob._realX - dx) * tw + this._shakeX);
            this.y = Math.round((baseY - dy + 1.3) * th + bobbing + this._shakeY);

            const mobScreenY = baseY;
            const playerY = $gamePlayer.y;
            this.z = (mobScreenY + 0.4 < playerY) ? 3 : 5;
        }


        updatePattern(pattern, direction) {
            if (!this._loaded) return;
            const dirIndex = { 2: 0, 4: 1, 6: 2, 8: 3 }[direction] ?? 0;
            const sx = this._sxBase + (pattern - 1) * this._pw;
            const sy = this._syBase + dirIndex * this._ph;
            this.setFrame(sx, sy, this._pw, this._ph);
        }

        updateDeathAnimation() {
            this._deathFrame++;
            const totalDuration = 60;

            this._offsetX += this._vx / 60;
            this._offsetY += this._vy / 60;
            this._vy += 0.1;

            this.rotation += this._rotationSpeed;

            // フェードアウト処理
            const fadeStart = totalDuration - 20;
            if (this._deathFrame > fadeStart) {
                const fadeProgress = (this._deathFrame - fadeStart) / 20;
                this.opacity = 255 * (1 - fadeProgress);
            }

            // Mob本体座標 + 吹き飛びオフセット
            const tw = $gameMap.tileWidth();
            const th = $gameMap.tileHeight();
            const dx = $gameMap.displayX();
            const dy = $gameMap.displayY();
            this.x = Math.round((this._mob._realX + this._offsetX - dx) * tw);
            this.y = Math.round((this._mob._realY + this._offsetY - dy) * th);

            if (this._deathFrame >= totalDuration) {
                if (this.parent) this.parent.removeChild(this);
                window.RSTH_IH._mobManager.removeMob(this._mob._id);
            }
        }



        startDeathAnimation() {
            this._isDying = true;
            this._deathFrame = 0;
            this._vx = (Math.random() - 0.5) * 6;
            this._vy = -4 - Math.random() * 10;
            this._rotationSpeed = (Math.random() - 0.5) * 0.5;

            this._offsetX = 0;  // 吹き飛び用オフセットを分離
            this._offsetY = 0;
        }


    };

    Sprite_Character.prototype.setSingleCharacterBitmap = function (bitmap) {
        this._characterName = bitmap._url || "";
        this._isBigCharacter = true;  // 1キャラは必ず大キャラ扱い
        this.bitmap = bitmap;
        this._characterIndex = 0; // 無視
    };

    // Mob用の武器攻撃関数
    window.RSTH_IH.useWeaponByMob = function (mob, weapon) {
        if (!weapon) return;

        const playerX = $gamePlayer.x;
        const playerY = $gamePlayer.y;

        const meta = weapon.meta;

        const knockback = Number(meta.knockback) || 0;

        const range = window.RSTH_IH.getWeaponRange(weapon);
        const shape = meta.shape ?? "box";

        for (let dx = -range; dx <= range; dx++) {
            for (let dy = -range; dy <= range; dy++) {
                const tx = playerX + dx;
                const ty = playerY + dy;

                const absDx = Math.abs(dx);
                const absDy = Math.abs(dy);

                const inRange = (shape === "circle") ? (absDx + absDy <= range) : true;

                if (inRange) {
                    if (tx !== playerX || ty !== playerY) continue;
                    if (!window.RSTH_IH.isTileReachable(mob.x(), mob.y(), tx, ty)) continue;

                    if ($gamePlayer.x === tx && $gamePlayer.y === ty) {
                        window.RSTH_IH.applyKnockbackToPlayer(mob, knockback);

                        window.RSTH_IH.damage_calculation(weapon, mob, mob, "mob", "actor");
                    }
                }
            }
        }
    };

    window.RSTH_IH.applyKnockbackToPlayer = function (attacker, knockback) {
        if (knockback <= 0) return;
        //console.log(`ConfigManager["gauge_KB"] `, ConfigManager["gauge_KB"]);
        if (ConfigManager["gauge_KB"] === 1) return;
        if ($gameParty.leader()?.hasState(29)) return;

        const centerX = attacker.x();
        const centerY = attacker.y();

        const offsetX = $gamePlayer.x - centerX;
        const offsetY = $gamePlayer.y - centerY;
        const length = Math.sqrt(offsetX * offsetX + offsetY * offsetY) || 0.0001;

        const normX = Math.round(offsetX / length);
        const normY = Math.round(offsetY / length);

        for (let i = 0; i < knockback; i++) {
            const nextX = $gamePlayer.x + normX;
            const nextY = $gamePlayer.y + normY;

            const dir = (normX === 1) ? 6 : (normX === -1) ? 4 : (normY === 1) ? 2 : (normY === -1) ? 8 : 0;

            // 🔽 マップ範囲と通行の両方をチェック
            if ($gameMap.isValid(nextX, nextY) && $gameMap.isPassable(nextX, nextY, dir)) {
                $gamePlayer._x = nextX;
                $gamePlayer._y = nextY;
                $gamePlayer._realX = nextX;
                $gamePlayer._realY = nextY;
            } else {
                break;
            }
        }
        window.RSTH_IH.requestSmoothCameraReset();

    };

    window.RSTH_IH._cameraResetScheduled = false;
    window.RSTH_IH._cameraResetWait = 0;
    window.RSTH_IH._cameraReturning = false;

    window.RSTH_IH.requestSmoothCameraReset = function () {
        if (window.RSTH_IH._cameraReturning) {
            // 既に戻してる途中なら再予約
            window.RSTH_IH._cameraResetScheduled = true;
            window.RSTH_IH._cameraResetWait = 60; // 1秒後
        } else {
            window.RSTH_IH._cameraResetScheduled = true;
            window.RSTH_IH._cameraResetWait = 60; // 1秒後
        }
    };

    Game_Player.prototype.updateScroll = function () {
        const marginX = (Graphics.width / $gameMap.tileWidth()) / 2;
        const marginY = (Graphics.height / $gameMap.tileHeight()) / 2;

        const targetX = this._realX - marginX;
        const targetY = this._realY - marginY;

        const currentX = $gameMap._displayX;
        const currentY = $gameMap._displayY;

        const diffX = targetX - currentX;
        const diffY = targetY - currentY;

        // カメラ戻しを予約中なら完全に静止（カメラ固定）
        if (window.RSTH_IH._cameraResetScheduled) {
            window.RSTH_IH._cameraResetWait--;

            const scrollThresholdX = 4; // タイル数で許容範囲
            const scrollThresholdY = 3;

            // プレイヤーが画面から大きく離れたら強制的にカメラ戻し開始
            if (Math.abs(diffX) > scrollThresholdX || Math.abs(diffY) > scrollThresholdY) {
                window.RSTH_IH._cameraResetScheduled = false;
                window.RSTH_IH._cameraReturning = true;
            } else if (window.RSTH_IH._cameraResetWait > 0) {
                return;
            } else {
                window.RSTH_IH._cameraResetScheduled = false;
                window.RSTH_IH._cameraReturning = true;
            }
        }



        // 戻し処理開始
        if (window.RSTH_IH._cameraResetWait <= 0 && window.RSTH_IH._cameraResetScheduled) {
            window.RSTH_IH._cameraResetScheduled = false;
            window.RSTH_IH._cameraReturning = true;
        }

        if (window.RSTH_IH._cameraReturning) {
            const smoothRate = 0.1;
            $gameMap._displayX += diffX * smoothRate;
            $gameMap._displayY += diffY * smoothRate;

            if (Math.abs(diffX) < 0.05 && Math.abs(diffY) < 0.05) {
                $gameMap._displayX = targetX;
                $gameMap._displayY = targetY;
                window.RSTH_IH._cameraReturning = false;
            }
        } else {
            // 通常の追従（ノックバック中は入らない）
            $gameMap._displayX = targetX;
            $gameMap._displayY = targetY;
        }

        const maxScrollX = $gameMap.width() - (Graphics.width / $gameMap.tileWidth());
        const maxScrollY = $gameMap.height() - (Graphics.height / $gameMap.tileHeight());

        $gameMap._displayX = Math.min(Math.max(0, $gameMap._displayX), maxScrollX);
        $gameMap._displayY = Math.min(Math.max(0, $gameMap._displayY), maxScrollY);
    };



    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
        _Scene_Map_start.call(this);

        if (!window.RSTH_IH._mobManager) {
            window.RSTH_IH._mobManager = new window.RSTH_IH.MobManager();
        } else {
            // ★ 既存のMobManagerがある場合はスプライトをTilemapに再付与
            window.RSTH_IH._mobManager.attachAllSprites();
        }
    };


    const _Scene_Map_terminate = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function () {
        if (window.RSTH_IH._mobManager) {
            window.RSTH_IH._mobManager.detachAllSprites();
        }

        _Scene_Map_terminate.call(this);
    };

    // MOB発生チェックメイン処理
    window.RSTH_IH.isStandableTile = function (x, y) {
        if (!$gameMap.isValid(x, y)) return false;
        if ([1, 7].includes($gameMap.terrainTag(x, y))) return false;
        if ($gameMap.eventsXy(x, y).length > 0) return false;
        if (window.RSTH_IH._mobManager.getMobAt(x, y)) return false;
        return [2, 4, 6, 8].some(dir => $gameMap.isPassable(x, y, dir));
    };

    // クエストでのmob発生チェック
    window.RSTH_IH.QuestMobSpawnCheck = function (count = 10) {
        // MAPのタイル数を読み込み
        const ax = $dataMap.width || 64;
        const ay = $dataMap.height || 64;

        // 初期値はマップ中央で障害物は考慮しない。
        let x = Math.round(ax / 2);
        let y = Math.round(ay / 2);
        for (let i = 0; i < count; i++) {
            x = Math.floor(Math.random() * ax);
            y = Math.floor(Math.random() * ay);

            if (!window.RSTH_IH.isStandableTile(x, y)) continue;
            return { x, y };
        }
        return { x, y };
    };

    window.RSTH_IH.AutoMobSpawner = {

        _spawnTimer: 60,  // 1秒分 (60FPSの場合)
        _maxMobs: window.RSTH_IH.MaxMobAmount,

        update: function () {
            //console.log("window.RSTH_IH.TimeCount", window.RSTH_IH.TimeCount);
            if (window.RSTH_IH.TimeCount >= window.RSTH_IH._elapsedLimit) return;

            if (--this._spawnTimer > 0) return; // ここで早期return

            this._spawnTimer = 10;

            const currentCount = window.RSTH_IH._mobManager._mobs.length;
            if (currentCount >= this._maxMobs) return;

            const rand = Math.random();
            const rand2 = Math.random();
            const area = rand < 0.5 ? "left" : "right";
            let x, y;

            let mob = null;
            let lankList = null;

            // map1
            if (window.RSTH_IH.AutoMobSpawnSet === 1) {
                // ▼ スポーン範囲①または②からランダムに選ぶ

                if (area === "left") {
                    x = 2;
                    y = Math.floor(rand * 6) + 26;
                } else {
                    x = 61;
                    y = Math.floor(rand * 7) + 36;
                }

                // 雑魚
                if ([300, 450, 750].includes(window.RSTH_IH.MobCount)) {
                    mob = window.RSTH_IH.spawnMob(8, x, y);
                } else if (window.RSTH_IH.MobCount < 751) {
                    lankList = window.RSTH_IH.MobLankTable.lank_1;
                } else if (window.RSTH_IH.MobCount < 1251) {
                    lankList = window.RSTH_IH.MobLankTable.lank_2;
                } else if (window.RSTH_IH.MobCount < 1800) {
                    lankList = window.RSTH_IH.MobLankTable.lank_3;
                }

                /*
                if ([300, 450, 750].includes(window.RSTH_IH.MobCount)) {
                    mob = window.RSTH_IH.spawnMob(8, x, y);
                } else if (window.RSTH_IH.TimeCount < 120) {
                    lankList = window.RSTH_IH.MobLankTable.lank_1;
                } else if (window.RSTH_IH.TimeCount < 210) {
                    lankList = window.RSTH_IH.MobLankTable.lank_2;
                } else if (window.RSTH_IH.TimeCount < 300) {
                    lankList = window.RSTH_IH.MobLankTable.lank_3;
                }
                */

                if (lankList && lankList.length > 0) {
                    const randIndex = Math.floor(rand * lankList.length);
                    const enemyId = (lankList[randIndex] - 1);
                    mob = window.RSTH_IH.spawnMob(enemyId, x, y);
                }

                window.RSTH_IH.MobCount++;

                if (mob) mob._aiMode = "chase";

            }

            else if (window.RSTH_IH.AutoMobSpawnSet === 2) {
                // ▼ スポーン範囲①または②からランダムに選ぶ

                if (area === "left") {
                    x = 2;
                    y = Math.floor(Math.random() * 57) + 29;
                } else {
                    x = 94;
                    y = Math.floor(Math.random() * 52) + 39;
                }

                if ([300, 450, 750, 900, 1200].includes(window.RSTH_IH.MobCount)) {
                    mob = window.RSTH_IH.spawnMob(8, x, y);
                }
                else if (window.RSTH_IH.MobCount < 501) lankList = window.RSTH_IH.MobLankTable.lank_1;
                else if (window.RSTH_IH.MobCount < 1201) lankList = window.RSTH_IH.MobLankTable.lank_2;
                else if (window.RSTH_IH.MobCount < 2100) lankList = window.RSTH_IH.MobLankTable.lank_3;
                else if (window.RSTH_IH.MobCount < 3000) lankList = window.RSTH_IH.MobLankTable.lank_4;
                else if (window.RSTH_IH.MobCount < 3600) lankList = window.RSTH_IH.MobLankTable.lank_5;


                if (lankList && lankList.length > 0) {
                    const randIndex = Math.floor(rand * lankList.length);
                    const enemyId = (lankList[randIndex] - 1);
                    mob = window.RSTH_IH.spawnMob(enemyId, x, y);
                }

                window.RSTH_IH.MobCount++;

                if (mob) mob._aiMode = "chase";

            }

            else if (window.RSTH_IH.AutoMobSpawnSet === 3) {
                // ▼ スポーン範囲①または②からランダムに選ぶ

                if (area === "left") {
                    x = Math.floor(rand * 4) + 45;
                    y = Math.floor(rand2 * 4) + 48;
                } else {
                    x = Math.floor(rand * 10) + 42;
                    y = Math.floor(rand2 * 2) + 2;
                }

                if ([300, 450, 750, 900, 1200].includes(window.RSTH_IH.MobCount)) {
                    mob = window.RSTH_IH.spawnMob(8, x, y);
                }
                else if (window.RSTH_IH.MobCount < 401) lankList = window.RSTH_IH.MobLankTable.lank_1;
                else if (window.RSTH_IH.MobCount < 801) lankList = window.RSTH_IH.MobLankTable.lank_2;
                else if (window.RSTH_IH.MobCount < 1400) lankList = window.RSTH_IH.MobLankTable.lank_3;
                else if (window.RSTH_IH.MobCount < 2200) lankList = window.RSTH_IH.MobLankTable.lank_4;
                else if (window.RSTH_IH.MobCount < 2900) lankList = window.RSTH_IH.MobLankTable.lank_5;
                else if (window.RSTH_IH.MobCount < 3600) lankList = window.RSTH_IH.MobLankTable.lank_6;
                else if (window.RSTH_IH.MobCount < 4000) mob = window.RSTH_IH.spawnMob(19, x, y);
                else if (window.RSTH_IH.MobCount < 4500) lankList = window.RSTH_IH.MobLankTable.lank_7;
                else if (window.RSTH_IH.MobCount < 5000) lankList = window.RSTH_IH.MobLankTable.lank_8;
                else if (window.RSTH_IH.MobCount < 5050) mob = window.RSTH_IH.spawnMob(24, x, y);


                if (lankList && lankList.length > 0) {
                    const randIndex = Math.floor(rand * lankList.length);
                    const enemyId = (lankList[randIndex] - 1);
                    mob = window.RSTH_IH.spawnMob(enemyId, x, y);
                }

                window.RSTH_IH.MobCount++;

                if (mob) mob._aiMode = "chase";

            }
            else if (window.RSTH_IH.AutoMobSpawnSet === 4) {
                // ▼ スポーン範囲①または②からランダムに選ぶ

                if (area === "left") {
                    x = Math.floor(rand * 5) + 2;
                    y = Math.floor(rand2 * 5) + 2;
                } else {
                    x = Math.floor(rand * 5) + 85;
                    y = Math.floor(rand2 * 5) + 88;
                }
                if ([30, 45, 75, 90, 120].includes(window.RSTH_IH.MobCount)) {
                    mob = window.RSTH_IH.spawnMob(8, x, y);
                }
                else if (window.RSTH_IH.MobCount === 905) mob = window.RSTH_IH.spawnMob(14, x, y);
                else if (window.RSTH_IH.MobCount === 1601) mob = window.RSTH_IH.spawnMob(19, x, y);
                else if (window.RSTH_IH.MobCount < 201) lankList = window.RSTH_IH.MobLankTable.lank_1;
                else if (window.RSTH_IH.MobCount < 401) lankList = window.RSTH_IH.MobLankTable.lank_2;
                else if (window.RSTH_IH.MobCount < 700) lankList = window.RSTH_IH.MobLankTable.lank_3;
                else if (window.RSTH_IH.MobCount < 1000) lankList = window.RSTH_IH.MobLankTable.lank_4;
                else if (window.RSTH_IH.MobCount < 1500) lankList = window.RSTH_IH.MobLankTable.lank_5;
                else if (window.RSTH_IH.MobCount < 2000) lankList = window.RSTH_IH.MobLankTable.lank_6;
                else if (window.RSTH_IH.MobCount < 2500) mob = window.RSTH_IH.spawnMob(19, x, y);
                else if (window.RSTH_IH.MobCount < 3000) lankList = window.RSTH_IH.MobLankTable.lank_7;
                else if (window.RSTH_IH.MobCount < 3500) lankList = window.RSTH_IH.MobLankTable.lank_8;
                else if (window.RSTH_IH.MobCount < 3510) mob = window.RSTH_IH.spawnMob(24, x, y);
                else if (window.RSTH_IH.MobCount < 3530) lankList = window.RSTH_IH.MobLankTable.lank_9;
                else if (window.RSTH_IH.MobCount < 3550) mob = window.RSTH_IH.spawnMob(24, x, y);


                if (lankList && lankList.length > 0) {
                    const randIndex = Math.floor(rand * lankList.length);
                    const enemyId = (lankList[randIndex] - 1);
                    mob = window.RSTH_IH.spawnMob(enemyId, x, y);
                }

                window.RSTH_IH.MobCount++;

                if (mob) mob._aiMode = "chase";

            }
            else if (window.RSTH_IH.AutoMobSpawnSet === 5) {
                // ▼ スポーン範囲①または②からランダムに選ぶ

                if (area === "left") {
                    x = Math.floor(rand * 8) + 1;
                    y = Math.floor(rand2 * 2) + 1;
                } else {
                    x = Math.floor(rand * 5) + 59;
                    y = Math.floor(rand2 * 5) + 59;
                }
                if ([10, 20, 30, 40, 50].includes(window.RSTH_IH.MobCount)) {
                    mob = window.RSTH_IH.spawnMob(8, x, y);
                }
                else if (window.RSTH_IH.MobCount === 60) mob = window.RSTH_IH.spawnMob(14, x, y);
                else if (window.RSTH_IH.MobCount === 70) mob = window.RSTH_IH.spawnMob(19, x, y);
                else if (window.RSTH_IH.MobCount < 201) lankList = window.RSTH_IH.MobLankTable.lank_5;
                else if (window.RSTH_IH.MobCount < 401) lankList = window.RSTH_IH.MobLankTable.lank_6;
                else if (window.RSTH_IH.MobCount < 601) mob = window.RSTH_IH.spawnMob(19, x, y);
                else if (window.RSTH_IH.MobCount < 1200) lankList = window.RSTH_IH.MobLankTable.lank_7;
                else if (window.RSTH_IH.MobCount < 2400) lankList = window.RSTH_IH.MobLankTable.lank_8;
                else if (window.RSTH_IH.MobCount < 2405) lankList = window.RSTH_IH.MobLankTable.lank_9;
                else if (window.RSTH_IH.MobCount < 2407) mob = window.RSTH_IH.spawnMob(24, x, y);
                else if (window.RSTH_IH.MobCount < 3600) lankList = window.RSTH_IH.MobLankTable.lank_10;


                if (lankList && lankList.length > 0) {
                    const randIndex = Math.floor(rand * lankList.length);
                    const enemyId = (lankList[randIndex] - 1);
                    mob = window.RSTH_IH.spawnMob(enemyId, x, y);
                }

                window.RSTH_IH.MobCount++;

                if (mob) mob._aiMode = "chase";

            }

        }
    };




})();
