/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */


(() => {
    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;


    //=============================================================================================================
    // ホットバー処理===============================================================================================
    //=============================================================================================================
    window.RSTH_IH.Window_Hotbar = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this.items = [];
            this.slotCount = window.RSTH_IH.HotbarSlotCount;
            this.tileSize = window.RSTH_IH.HotbarSlotSize;
            this.margin = window.RSTH_IH.Hotbarmargin;
            if (window.RSTH_IH.HobarSlotsIndex === -1) window.RSTH_IH.HobarSlotsIndex = 0;
            this.selectedIndex = window.RSTH_IH.HobarSlotsIndex;
            this.select(window.RSTH_IH.HobarSlotsIndex);
            this._lastClickTime = null;
            this._lastClickIndex = null;
            this.hoverCache = new Array(this.slotCount).fill(null);

            for (let i = 0; i < this.items.length; i++) {
                this.hoverCache[i] = this.items[i];
            }

            this.refresh();
            this.setupHotbarKeyListener();

            this._mouseHandlersSet = true;
        }

        processCursorMove() { }

        maxItems() {
            return this.slotCount;
        }

        maxCols() {
            return this.slotCount;
        }

        // 行数を強制設定(1)
        maxRows() {
            return 1;
        }

        maxScrollY() {
            return 0; // スクロール不可にする
        }

        itemRect(index) {
            const totalContentWidth = this.slotCount * this.tileSize + (this.slotCount - 1) * this.margin;
            const startX = (this.contentsWidth() - totalContentWidth) / 2;
            const x = startX + index * (this.tileSize + this.margin);
            const y = this.margin / 2;
            return new Rectangle(x, y, this.tileSize, this.tileSize);
        }

        refresh() {
            if (RSTH_DEBUG_LOG) console.log("[Window_Hotbar]refresh描画開始");
            this.items = $gameSystem._customHotbarItems || [];
            window.RSTH_IH.drawItemSlotGrid(this, this.items, this.slotCount, 1, this.tileSize, this.margin, this.selectedIndex, true);
            this.hoverCache = [...this.items];
            if (SceneManager._scene?._rsthCursorSprite && SceneManager._scene.updateCursorForSlot && this.selectedIndex >= 0) {
                SceneManager._scene.updateCursorForSlot(this.selectedIndex, this);
            }

            for (let i = 0; i < this.slotCount; i++) {
                const rect = this.itemRect(i);
                const cooldown = window.RSTH_IH._cooldownMap?.[i];
                if (cooldown && cooldown.remaining > 0) {
                    const progress = cooldown.remaining / cooldown.duration;
                    this.drawCooldownOverlay(rect, progress);
                }
            }


        }

        drawCooldownOverlay(rect, progress) {
            const ctx = this.contents.context;
            const cx = rect.x + rect.width / 2;
            const cy = rect.y + rect.height / 2;
            const radius = rect.width / 2 - 4;
            const start = -Math.PI / 2;
            const end = start + Math.PI * 2 * progress;

            ctx.save();
            ctx.beginPath();
            ctx.moveTo(cx, cy);
            ctx.arc(cx, cy, radius, start, end, false);
            ctx.closePath();
            ctx.fillStyle = 'rgba(0, 0, 0, 0.8)';
            ctx.fill();
            ctx.restore();
        }


        canvasToLocal(event) {
            const bounds = event.target.getBoundingClientRect();
            return {
                x: event.clientX - bounds.left,
                y: event.clientY - bounds.top
            };
        }

        hitTest(x, y) {
            for (let i = 0; i < this.maxItems(); i++) {
                const rect = this.itemRect(i);
                if (x >= rect.x && x < rect.x + rect.width && y >= rect.y && y < rect.y + rect.height) {
                    return i;
                }
            }
            return -1;
        }

        select(index) {
            this.selectedIndex = index;
            window.RSTH_IH.HobarSlotsIndex = index;
            window.RSTH_IH.DirectedDoor = 0;

            //  抑制されているときは再描画だけ、使用処理は走らせない
            if (!this._suppressUse) {
                this.refresh();
            }
        }

        selectSlotByIndex(index) {
            if (index >= 0 && index < this.slotCount) {
                this.selectedIndex = index;
                this.refresh();
            }
        }

        selectNextSlot() {
            this.selectedIndex = (this.selectedIndex + 1) % this.slotCount;
            window.RSTH_IH.HobarSlotsIndex = this.selectedIndex;
            window.RSTH_IH.DirectedDoor = 0;
            this.refresh();
        }

        selectPreviousSlot() {
            this.selectedIndex = (this.selectedIndex - 1 + this.slotCount) % this.slotCount;
            window.RSTH_IH.HobarSlotsIndex = this.selectedIndex;
            window.RSTH_IH.DirectedDoor = 0;
            this.refresh();
        }

        setupHotbarKeyListener() {
            if (window.RSTH_IH.__hotbarKeyListenerAdded) return;
            window.RSTH_IH.__hotbarKeyListenerAdded = true;

            document.addEventListener("keydown", (event) => {
                const scene = SceneManager._scene;
                if (!(scene && scene._hotbarWindow)) return;

                const key = event.key;
                const hotbarWindow = scene._hotbarWindow;

                if (key >= "1" && key <= "9") {
                    const index = Number(key) - 1;
                    hotbarWindow.selectSlotByIndex(index);
                    const item = hotbarWindow.items[index];
                    hotbarWindow.setItems($gameSystem._customHotbarItems);
                    window.RSTH_IH.HobarSlotsIndex = index;
                    if (item) {
                        window.RSTH_IH.DirectedDoor = 0;
                    }
                    this.useItemsAtHotbarKey(item, index);

                } else if (key === "0") {
                    const index = 9;
                    hotbarWindow.selectSlotByIndex(index);
                    const item = hotbarWindow.items[index];
                    hotbarWindow.setItems($gameSystem._customHotbarItems);

                    window.RSTH_IH.HobarSlotsIndex = index;
                    if (item) {
                        window.RSTH_IH.DirectedDoor = 0;
                    }
                    this.useItemsAtHotbarKey(item, index);

                } else if (key === window.RSTH_IH.HotbarPrevKey) {
                    hotbarWindow.selectPreviousSlot();
                } else if (key === window.RSTH_IH.HotbarNextKey) {
                    hotbarWindow.selectNextSlot();
                }
            });
        }

        useItemsAtHotbarKey(item, index) {
            // ホットバーで選択されているアイテムを使用する
            if (SceneManager._scene instanceof Scene_Map) {
                //console.log("数字キーでホットバー使用を試行");
                //console.log("item", item);
                //console.log("index", index);

                if (item) {
                    if (item.type === "weapon") {
                        //console.log(`ホットバーのカーソル index ${index} の武器を使用:`, item);
                        window.RSTH_IH.onLeftButtonDown(true);
                        return;
                    } else if (item.type === "item") {
                        //console.log(`ホットバーのカーソル index ${index} のアイテムを使用:`, item);
                        window.RSTH_IH.useInventoryItem(item, "hotbar", index);
                        return;

                    } else {
                        return;
                    }
                } else {
                    if (RSTH_DEBUG_LOG) console.log(`カーソル index ${index} にアイテムが存在しません`);
                }
            }
        }

        processTouch() {
            super.processTouch();

            window.RSTH_IH.handleSlotTouchShared(this, "hotbar", this.items, (item, index) => {
                window.RSTH_IH.useInventoryItem(item, "hotbar", index);
            });


            if (window.RSTH_IH.DD) {
                window.RSTH_IH.handleInventoryDragDrop(this);
            }
        }

        canvasToLocalX(globalX) {
            return globalX - this.x - this.padding;
        }

        canvasToLocalY(globalY) {
            return globalY - this.y - this.padding;
        }

        setItems(items) {
            window.RSTH_IH.setItemsSafe(this, items, "_customHotbarItems");
            if (RSTH_DEBUG_LOG) console.log("[Window_Hotbar]setItemsSafe実行");

            this.refresh();
        }

        update() {
            Window_Selectable.prototype.update.call(this);
            this.updateHoverText();
            this.updateWheelScroll();

        }

        updateWheelScroll() {
            if (!this.active || !this.visible || !this.open) return;

            const scene = SceneManager._scene;
            if (!(scene && scene._hotbarWindow)) return;
            const threshold = 1;
            const dy = TouchInput.wheelY;
            const hotbarWindow = scene._hotbarWindow;

            if (dy >= threshold) {
                hotbarWindow.selectPreviousSlot();
                TouchInput._wheelY = 0;
            } else if (dy <= -threshold) {
                hotbarWindow.selectNextSlot();
                TouchInput._wheelY = 0;
            }
        }


        updateHoverText() {
            window.RSTH_IH.updateHoverTextShared(this, this.items);
        }
    }

    //=============================================================================================================
    // インベントリ処理===============================================================================================
    //=============================================================================================================
    window.RSTH_IH.Window_Inventory = class extends Window_Selectable {
        // コンストラクタ内でパディングを定義
        constructor(rect) {
            super(rect);
            this.items = [];
            this.tileSize = window.RSTH_IH.InventorySlotSize;
            this.margin = window.RSTH_IH.Inventorymargin;  // アイテム間のマージン
            this._lastClickTime = null;
            this._lastClickIndex = null;
            this.hoverCache = new Array(this.maxItems()).fill(null);

            for (let i = 0; i < this.items.length; i++) {
                this.hoverCache[i] = this.items[i];
            }

            this.refresh();

            this._mouseHandlersSet = true;
            this.select(-1); // 選択を無効化（カーソル非表示）

        }

        // アイテムの数を固定
        maxItems() {
            const maxItemsSize = window.RSTH_IH.InventoryCols * window.RSTH_IH.InventoryRows;
            return maxItemsSize;
        }

        // 列数を強制設定（9列）
        maxCols() {
            return window.RSTH_IH.InventoryCols;
        }

        // 行数を強制設定（4行）
        maxRows() {
            return window.RSTH_IH.InventoryRows;
        }

        maxScrollY() {
            return 0; // スクロール不可にする
        }

        // アイテムの位置計算
        itemRect(index) {
            const col = index % this.maxCols();
            const row = Math.floor(index / this.maxCols());
            const totalContentWidth = this.maxCols() * this.tileSize + (this.maxCols() - 1) * this.margin;
            const startX = (this.contentsWidth() - totalContentWidth) / 2;
            const x = startX + col * (this.tileSize + this.margin);
            const y = row * (this.tileSize + this.margin) + this.margin / 2;
            return new Rectangle(x, y, this.tileSize, this.tileSize);
        }


        // リフレッシュ
        refresh() {
            if (RSTH_DEBUG_LOG) console.log("[Window_Inventory]refresh描画開始");
            this.items = $gameSystem._customInventoryItems || [];
            window.RSTH_IH.drawItemSlotGrid(this, this.items, window.RSTH_IH.InventoryCols, window.RSTH_IH.InventoryRows, this.tileSize, this.margin, this.selectedIndex);
            this.hoverCache = [...this.items];

            const scene = SceneManager._scene;
            const sprite = scene?._rsthCursorSprite;

            if (sprite) {
                if (!this.visible || this.openness <= 0 || this.selectedIndex < 0) {
                    // 非表示条件：ウィンドウ非表示、閉じかけ、選択なし
                    sprite.visible = false;
                } else if (scene.updateCursorForSlot) {
                    // 通常更新
                    scene.updateCursorForSlot(this.selectedIndex, this);
                }
            }


        }

        // 座標補正
        canvasToLocal(event) {
            const bounds = event.target.getBoundingClientRect();
            return {
                x: event.clientX - bounds.left,
                y: event.clientY - bounds.top
            };
        }

        hitTest(x, y) {
            for (let i = 0; i < this.maxItems(); i++) {
                const rect = this.itemRect(i);
                if (x >= rect.x && x < rect.x + rect.width && y >= rect.y && y < rect.y + rect.height) {
                    return i;
                }
            }
            return -1;
        }

        select(index) {
            this.selectedIndex = index;

            //  抑制されているときは再描画だけ、使用処理は走らせない
            if (!this._suppressUse) {
                this.refresh();
            }
        }

        selectSlotByIndex(index) {
            if (index >= 0 && index < this.maxItems()) {
                this.selectedIndex = index;
                this.refresh();
            }
        }

        processTouch() {
            if (!this.visible || this.openness <= 0) return;
            super.processTouch();

            window.RSTH_IH.handleSlotTouchShared(this, "inventory", this.items, (item, index) => {
                window.RSTH_IH.useInventoryItem(item, "inventory", index);
            });

        }

        processCursorMove() { }

        canvasToLocalX(globalX) {
            return globalX - this.x - this.padding;
        }

        canvasToLocalY(globalY) {
            return globalY - this.y - this.padding;
        }

        setItems(items) {
            window.RSTH_IH.setItemsSafe(this, items, "_customInventoryItems");
            if (RSTH_DEBUG_LOG) console.log("[Window_Inventory]setItemsSafe実行");
            this.refresh();
        }

        update() {
            Window_Selectable.prototype.update.call(this);
            this.updateHoverText();
        }

        updateHoverText() {
            window.RSTH_IH.updateHoverTextShared(this, this.items);
        }
    }


    window.RSTH_IH.Window_Inventory.prototype.hide = function () {
        Window_Selectable.prototype.hide.call(this);

        if (window.RSTH_IH.__popupOwner === this) {
            const popup = SceneManager._scene._hoverTextSprite;
            if (popup) popup.setText("");
            window.RSTH_IH.__popupOwner = null;
        }
        if (SceneManager._scene?._rsthCursorSprite) {
            SceneManager._scene._rsthCursorSprite.visible = false;
        }

    };


    //=============================================================================================================
    // 装備ウィンドウ処理===============================================================================================
    //=============================================================================================================

    window.RSTH_IH.Window_EquipmentSlots = class extends Window_Selectable {
        constructor(rect) {
            super(rect);

            this._actor = $gameParty.leader();
            this.tileSize = window.RSTH_IH.EQUIP_SLOT_SIZE;
            this.margin = window.RSTH_IH.Eqslotmargin;

            this.selectedIndex = 0;
            // maxCols() 発火前に this.slotIndices を必ず定義済みにする
            this.slotIndices = [];

            // 装備スロット構成を動的に取得（武器は除外）
            this.slotIndices = this._actor.equipSlots()
                .map((etypeId, i) => ({ etypeId, index: i }))
                .filter(obj => obj.etypeId !== 1)
                .map(obj => obj.index);

            this.items = this.slotIndices.map(slotId => this._actor.equips()[slotId]);
            this._lastClickIndex = null;
            this._lastClickTime = 0;
            this._clickIndex = -1;
            this.hoverCache = [...this.items];



            this.refresh();
        }


        processCursorMove() { }

        maxItems() {
            return this.slotIndices.length;
        }

        maxCols() {
            return (this.slotIndices?.length || 0);
        }

        maxRows() {
            return 1;
        }

        maxScrollY() {
            return 0;
        }

        itemRect(index) {
            const totalWidth = this.items.length * this.tileSize + (this.items.length - 1) * this.margin;
            const startX = (this.contentsWidth() - totalWidth) / 2;
            const x = startX + index * (this.tileSize + this.margin);
            const y = this.margin / 2;
            return new Rectangle(x, y, this.tileSize, this.tileSize);
        }

        refresh() {
            this.contents.clear();

            this.slotIndices = this._actor.equipSlots()
                .map((etypeId, i) => ({ etypeId, index: i }))
                .filter(obj => obj.etypeId !== 1)
                .map(obj => obj.index);

            this.items = this.slotIndices.map(slotId => this._actor.equips()[slotId]);
            this.hoverCache = [...this.items]; // ← 安定のため再構築

            const totalWidth = this.slotIndices.length * this.tileSize + (this.slotIndices.length - 1) * this.margin;
            const totalHeight = this.tileSize;
            this.width = totalWidth + this.padding * 3;
            this.height = totalHeight + this.padding * 2 + this.margin;
            this.createContents();

            for (let i = 0; i < this.maxItems(); i++) {
                this.drawItem(i);
            }
        }


        canvasToLocal(event) {
            const bounds = event.target.getBoundingClientRect();
            return {
                x: event.clientX - bounds.left,
                y: event.clientY - bounds.top
            };
        }


        hitTest(x, y) {
            for (let i = 0; i < this.maxItems(); i++) {
                const rect = this.itemRect(i);
                if (x >= rect.x && x < rect.x + rect.width && y >= rect.y && y < rect.y + rect.height) {
                    return i;
                }
            }
            return -1;
        }

        select(index) {
            this.selectedIndex = index;
        }

        selectSlotByIndex(index) {
            if (index >= 0 && index < this.slotIndices.length) {
                this.selectedIndex = index;
                this.refresh();
            }
        }


        processTouch() {
            /*
            if (TouchInput.isTriggered()) {
                const x = this.canvasToLocalX(TouchInput.x);
                const y = this.canvasToLocalY(TouchInput.y);
                this._clickIndex = this.hitTest(x, y);
                this._clickTime = performance.now();
            }

            if (TouchInput.isReleased() && this._clickIndex >= 0) {
                const x = this.canvasToLocalX(TouchInput.x);
                const y = this.canvasToLocalY(TouchInput.y);
                const index = this.hitTest(x, y);
                if (index === this._clickIndex) {
                    const now = performance.now();
                    if (this._lastClickTime && now - this._lastClickTime < 300 && index === this._lastClickIndex) {
                        this.onDoubleClick(index);
                        this._lastClickTime = 0;
                        this._lastClickIndex = -1;
                    } else {
                        this._lastClickTime = now;
                        this._lastClickIndex = index;
                    }
                }
                this._clickIndex = -1;
            }
                */
        }

        canvasToLocalX(globalX) {
            const realX = this.absoluteX(); // ウィンドウの画面内X
            return globalX - realX - this.padding;
        }

        canvasToLocalY(globalY) {
            const realY = this.absoluteY(); // ウィンドウの画面内Y
            return globalY - realY - this.padding;
        }

        // 追加：画面内の絶対位置を取得
        absoluteX() {
            let x = this.x;
            let parent = this.parent;
            while (parent) {
                x += parent.x || 0;
                parent = parent.parent;
            }
            return x;
        }

        absoluteY() {
            let y = this.y;
            let parent = this.parent;
            while (parent) {
                y += parent.y || 0;
                parent = parent.parent;
            }
            return y;
        }






        drawItem(index) {
            const rect = this.itemRect(index);
            this.contents.strokeRect(rect.x, rect.y, rect.width, rect.height, "white");

            const item = this.items[index];
            if (item) {
                const iconSet = ImageManager.loadSystem("IconSet");
                const pw = 32, ph = 32;
                const sx = item.iconIndex % 16 * pw;
                const sy = Math.floor(item.iconIndex / 16) * ph;

                const scale = this.tileSize / 32;
                const dw = pw * scale;
                const dh = ph * scale;
                const dx = rect.x + (rect.width - dw) / 2;
                const dy = rect.y + (rect.height - dh) / 2;

                this.contents.blt(iconSet, sx, sy, pw, ph, dx, dy, dw, dh);
            }
            this.hoverCache[index] = item;

        }


        onDoubleClick(index) {
            const slotId = this.slotIndices[index];
            const actor = this._actor;
            const item = actor.equips()[slotId];
            if (!item) return;

            const dbItem = DataManager.isArmor(item) ? $dataArmors[item.id]
                : DataManager.isWeapon(item) ? $dataWeapons[item.id]
                    : null;
            if (!dbItem) return;

            if (!window.RSTH_IH.hasFreeSpaceForItem(dbItem)) {
                SoundManager.playBuzzer();
                if (RSTH_DEBUG_LOG) console.warn("[RSTH] 装備解除失敗：収納先が満杯");
                return;
            }

            actor.changeEquip(slotId, null);
            SoundManager.playEquip();
            this.refresh();

            if (SceneManager._scene?.updateInventoryAndHotbar) {
                SceneManager._scene.updateInventoryAndHotbar();
            }
        }


        update() {
            Window_Selectable.prototype.update.call(this);
            this.hoverCache = [...this.items];

            this.updateHoverText();
        }

        updateHoverText() {
            //console.log("Hover Pos:", TouchInput.x, TouchInput.y);
            //console.log("Local Pos:", this.canvasToLocalX(TouchInput.x), this.canvasToLocalY(TouchInput.y));
            //console.log("Abs Pos:", this.absoluteX(), this.absoluteY());

            window.RSTH_IH.updateHoverTextShared(this, this.hoverCache);
        }







    }




})();