/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 * 
 * @param HotbarPosition
 * @type select
 * @option topleft
 * @option top
 * @option topright
 * @option bottomleft
 * @option bottom
 * @option bottomright
 * @default bottomright
 * @desc ホットバーとインベントリの画面上の位置を指定します。
 * 
 * @param HotbarSlotSize
 * @type number
 * @default 32
 * @desc ホットバースロットの1マスのサイズ（ピクセル単位）
 *
 * @param HotbarPrevKey
 * @type string
 * @default [
 * @desc ホットバーで前のスロットを選択するキー
 *
 * @param HotbarNextKey
 * @type string
 * @default ]
 * @desc ホットバーで次のスロットを選択するキー
 *
 * @param InventoryCols
 * @type number
 * @default 10
 * @desc インベントリの1行のスロット数
 *
 * @param InventoryRows
 * @type number
 * @default 6
 * @desc インベントリの最大行数
 *
 * @param InventorySlotSize
 * @type number
 * @default 32
 * @desc インベントリスロットの1マスのサイズ（ピクセル単位）
 *
 * @param gainItemToInventoryThenHotbar
 * @type boolean
 * @default true
 * @desc アイテムを入手時、先にインベントリに入れるならON。
 * 
 * @param toggleInventoryKey
 * @type boolean
 * @default true
 * @desc インベントリウィンドウを開かせるならON。
 * 
 * @param StackSize
 * @type number
 * @default 99
 * @desc 各スロットに格納できる最大数
 * 
 * @param pickupRange
 * @type number
 * @default 3
 * @desc ドロップアイテムを回収できる最大距離
 * 
 * @param growBlock
 * @type string
 * @default plant
 * @desc 成長できるブロックタイプ
 * 
 * @param EnableWeaponEquip
 * @type boolean
 * @default true
 * @desc ダブルクリックで武器を装備する機能をON/OFFできます。
 * 
 * @param EquipmentUI_SLOT_SIZE
 * @type number
 * @default 32
 * @desc 装備スロットの1マスのサイズ（ピクセル単位）

 * @param EquipmentUI_EQUIP_INDICES
 * @type number[]
 * @default ["1","2","3","4"]
 * @desc 対象とする装備スロットの番号配列（武器=0、盾=1、頭=2...）
 *
 * @param EquipmentUI_Position
 * @type select
 * @option topleft
 * @option topright
 * @option bottomleft
 * @option bottomright
 * @option ori
 * @default bottomleft
 * @desc 装備ウィンドウを表示する画面の位置
 * 
 * @param EquipmentUI_ToggleKey
 * @type string
 * @default e
 * @desc 装備ウィンドウの表示／非表示を切り替えるキー（小文字）
 * 
 * @param DD
 * @type boolean
 * @default true
 * @desc D&D有効か否か。ONで有効。
 * 
 * @param AutotileShapeList
 * @text オートタイルShape定義リスト
 * @type struct<AutotileShapeSet>[]
 * @desc 使用するオートタイルShape画像を登録
 *
 * @help
 * img/tilesets フォルダに置いた各オートタイルshape画像 (floor_tileId85_Inside_A4 など)
 * を複数登録できます。登録した分だけキャッシュされます。
 * 
 * @param TilesetConfigs
 * @type struct<TilesetConfig>[]
 * @default []
 * @desc タイルセットごとの tileSize や cols 設定
 * 
 * @param MobD_mzonoff
 * @type boolean
 * @default true
 * @desc Mob歩行キャラの画像の読込形式をMZ形式とするか。ONでMZ形式。
 * 
 * @param NpcD_mzonoff
 * @type boolean
 * @default true
 * @desc Npc歩行キャラの画像の読込形式をMZ形式とするか。ONでMZ形式。
 * 
 * @param DmgPopup
 * @type boolean
 * @default true
 * @desc ダメージポップアップをするか否か。ONでする。
 * 
 * @param MobDefinitions
 * @text Mob定義一覧
 * @type struct<MobDefinition>[]
 * @desc Mobの種別定義リスト
 * 
 * @param NpcDefinitions
 * @text Npc定義一覧
 * @type struct<NpcDefinition>[]
 * @desc Npcの種別定義リスト
 */

/*~struct~TilesetConfig:
 * @param name
 * @text タイルセット名
 * @desc タイル画像ファイル名（拡張子不要）
 * 
 * @param tileSize
 * @text タイルサイズ
 * @type number
 * @default 48
 *
 * @param cols
 * @text 列数
 * @type number
 * @default 16
 * 
 */

/*~struct~AutotileShapeSet:
 * @param TilesetName
 * @text 対象タイルセット画像
 * @type file
 * @dir img/tilesets
 * @desc オートタイルShapeを適用するタイルセット画像
 */

/*~struct~MobDefinition:
 * 
 * @param name
 * @text Mob名
 * @type string
 * @desc Mobの名前を指定します
 * 
 * @param isEnemy
 * @text 敵フラグ
 * @type boolean
 * @desc trueで敵、falseで味方
 * 
 * @param databaseId
 * @text データベースID
 * @type number
 * @desc 参照するデータベースのID（敵またはアクターのID）
 * 
 * @param characterName
 * @text 歩行キャラ画像
 * @type file
 * @dir img/characters
 * @desc 使用する歩行キャラ画像を指定
 * 
 * @param characterIndex
 * @text キャラ位置Index
 * @type number
 * @min 0
 * @max 7
 * @desc 画像内のキャラ位置 (0～7)
 * 
 * @param defaultLv
 * @text 初期レベル
 * @type number
 * @min 1
 * @default 1
 * @desc 初期レベル
 * 
 * @param defaultExp
 * @text 初期経験値
 * @type number
 * @min 0
 * @default 0
 * @desc 初期経験値
 * 
 * @param moveSpd
 * @text 移動速度
 * @type number
 * @min 1
 * @default 1
 * @desc 移動速度
 * 
 */

/*~struct~NpcDefinition:
 * 
 * @param name
 * @text Npc名
 * @type string
 * @desc Npcの名前を指定します
 * 
 * @param isEnemy
 * @text 敵フラグ
 * @type boolean
 * @desc trueで敵、falseで味方
 * 
 * @param databaseId
 * @text データベースID
 * @type number
 * @desc 参照するデータベースのID（敵またはアクターのID）
 * 
 * @param characterImg
 * @text 顏画像
 * @type file
 * @dir img/faces
 * @desc 使用する顏画像を指定
 * 
 * @param faceIndex
 * @text 顔画像Index
 * @type number
 * @min 0
 * @max 7
 * @desc 顔画像内の位置 (0～7)
 * 
 * @param characterName
 * @text 歩行キャラ画像
 * @type file
 * @dir img/characters
 * @desc 使用する歩行キャラ画像を指定
 * 
 * @param characterIndex
 * @text キャラ位置Index
 * @type number
 * @min 0
 * @max 7
 * @desc 画像内のキャラ位置 (0～7)
 * 
 * @param defaultLv
 * @text 初期レベル
 * @type number
 * @min 1
 * @default 1
 * @desc 初期レベル
 * 
 * @param defaultExp
 * @text 初期経験値
 * @type number
 * @min 0
 * @default 0
 * @desc 初期経験値
 * 
 * @param moveSpd
 * @text 移動速度
 * @type number
 * @min 1
 * @default 1
 * @desc 移動速度
 * 
 */


(() => {
  "use strict";

  window.RSTH_IH = window.RSTH_IH || {};

  // ログ出力制御フラグ（trueでログ出力、falseで抑制）
  //const RSTH_DEBUG_LOG = true;
  const RSTH_DEBUG_LOG = false;

  document.addEventListener("contextmenu", function (event) {
    event.preventDefault();
    return false;
  });

  window.RSTH_IH.__rsth_shouldRunOnMapLoaded = true; // メニュー開閉後判定用

  const filename = document.currentScript.src.match(/([^\/]+)\.js$/)[1];  //拡張子なしのファイル名を参照
  window.RSTH_IH.parameters = PluginManager.parameters(filename);

  window.RSTH_IH.StackSize = Number(window.RSTH_IH.parameters["StackSize"] || 99);

  //ホットバー関連の宣言
  window.RSTH_IH.HotbarSlotSize = Number(window.RSTH_IH.parameters["HotbarSlotSize"] || 32);
  window.RSTH_IH.HotbarPrevKey = window.RSTH_IH.parameters["HotbarPrevKey"] || "[";
  window.RSTH_IH.HotbarNextKey = window.RSTH_IH.parameters["HotbarNextKey"] || "]";
  window.RSTH_IH.HotbarSlotCount = 10;//10固定。変更しないこと
  window.RSTH_IH.Hotbarmargin = 8;
  window.RSTH_IH.Hotbarpadding = 12;
  window.RSTH_IH.HotbarcontentWidth = window.RSTH_IH.HotbarSlotCount * window.RSTH_IH.HotbarSlotSize + (window.RSTH_IH.HotbarSlotCount - 1) * window.RSTH_IH.Hotbarmargin;
  window.RSTH_IH.Hotbarwidth = window.RSTH_IH.HotbarcontentWidth + window.RSTH_IH.Hotbarpadding * 2 + window.RSTH_IH.Hotbarmargin;
  window.RSTH_IH.Hotbarheight = window.RSTH_IH.HotbarSlotSize + window.RSTH_IH.Hotbarpadding * 2 + window.RSTH_IH.Hotbarmargin;
  window.RSTH_IH.HotbarPosition = window.RSTH_IH.parameters["HotbarPosition"] || "bottomright";

  //インベントリ関連の宣言
  window.RSTH_IH.InventorySlotSize = Number(window.RSTH_IH.parameters["InventorySlotSize"] || 32);
  window.RSTH_IH.InventoryCols = Number(window.RSTH_IH.parameters["InventoryCols"] || 10);
  window.RSTH_IH.InventoryRows = Number(window.RSTH_IH.parameters["InventoryRows"] || 6);
  window.RSTH_IH.Inventorymargin = 8;
  window.RSTH_IH.Inventorypadding = 12;
  window.RSTH_IH.toggleInventoryKey = window.RSTH_IH.parameters["toggleInventoryKey"] === "true";

  window.RSTH_IH.InventorycontentWidth = window.RSTH_IH.InventoryCols * window.RSTH_IH.InventorySlotSize + (window.RSTH_IH.InventoryCols) * window.RSTH_IH.Inventorymargin;
  window.RSTH_IH.InventorycontentHeight = window.RSTH_IH.InventoryRows * window.RSTH_IH.InventorySlotSize + (window.RSTH_IH.InventoryRows) * window.RSTH_IH.Inventorymargin;

  window.RSTH_IH.Inventorywidth = window.RSTH_IH.InventorycontentWidth + window.RSTH_IH.Inventorypadding * 2;
  window.RSTH_IH.Inventoryheight = window.RSTH_IH.InventorycontentHeight + window.RSTH_IH.Inventorypadding * 2;

  window.RSTH_IH.getItemToInventoryThenHotbar = window.RSTH_IH.parameters["gainItemToInventoryThenHotbar"] === "true";

  window.RSTH_IH.SetShop = 1;
  window.RSTH_IH.SetTreasure = 1;

  window.RSTH_IH.StatusWindowWidth = 0;
  window.RSTH_IH.StatusWindowHeight = 0;
  window.RSTH_IH.StatusSubWindowWidth = 0;
  window.RSTH_IH.StatusSubWindowHeight = 0;

  window.RSTH_IH.EnableWeaponEquip = window.RSTH_IH.parameters["EnableWeaponEquip"] === "true";

  window.RSTH_IH.DD = window.RSTH_IH.parameters["DD"] === "true";

  window.RSTH_IH._calledFromSelectMap = false;

  window.RSTH_IH.PickUpRange = Number(window.RSTH_IH.parameters["pickupRange"] || 3);

  window.RSTH_IH.GrowBlock = window.RSTH_IH.parameters["growBlock"] || "plant";

  window.RSTH_IH.__draggingItem = null;
  window.RSTH_IH.__draggingFrom = null;
  window.RSTH_IH.__draggingIndex = null;

  window.RSTH_IH.__popupOwner = null;

  window.RSTH_IH.__hotbarKeyListenerAdded = false;

  // ホットバーのカーソル位置保存用
  window.RSTH_IH.HobarSlotsIndex = 0;

  // 選択したマップID保存用
  window.RSTH_IH._selectMapId = 0;
  window.RSTH_IH.getUnshoP = Number(0);

  // ショップ用変数
  window.RSTH_IH.BuyShop = [];

  window.RSTH_IH.__needGrowthSpriteUpdate = false;

  window.RSTH_IH.TILESET_NAME = window.RSTH_IH.parameters["TilesetName"] || "Inside_C";

  window.RSTH_IH.tilesetConfigsRaw = window.RSTH_IH.parameters["TilesetConfigs"] || "[]";
  if (RSTH_DEBUG_LOG) console.warn("tilesetConfigsRaw", window.RSTH_IH.tilesetConfigsRaw);

  window.RSTH_IH.DirectedDoor = 0;

  window.RSTH_IH.EQUIP_SLOT_SIZE = Number(window.RSTH_IH.parameters["EquipmentUI_SLOT_SIZE"] || 32); // スロット1つのサイズ
  window.RSTH_IH.EQUIP_POSITION = window.RSTH_IH.parameters["EquipmentUI_Position"] || "bottomleft";
  window.RSTH_IH.EQUIP_TOGGLE_KEY = (window.RSTH_IH.parameters["EquipmentUI_ToggleKey"] || "e").toLowerCase();

  // 装備スロット配列の変換（string[] → number[]）
  window.RSTH_IH.EQUIP_INDICES = JSON.parse(window.RSTH_IH.parameters["EquipmentUI_EQUIP_INDICES"] || "[1,2,3,4]").map(Number);

  window.RSTH_IH.Eqslotmargin = 8;

  // 初期化時に指定キーを仮想アクション "toggleEquipment" に割り当て
  Input.keyMapper = Input.keyMapper || {};
  Input.keyMapper[window.RSTH_IH.EQUIP_TOGGLE_KEY.toUpperCase().charCodeAt(0)] = "toggleEquipment";

  Input.keyMapper[82] = 'R';  // 82はRキーのkeyCode
  Input.keyMapper = Object.assign(Input.keyMapper, {
    73: 'i' // キーコード73は 'I' キー
  });
  Input.keyMapper[84] = "t"; // 84 は Tキーの keyCode
  Input.keyMapper[79] = "o"; // Oキーを "o" として登録
  Input.keyMapper[69] = "e";
  Input.keyMapper[76] = "l";

  window.RSTH_IH.BossRush = false;

  // 初期選択マップ
  window.RSTH_IH.MapName = "イテルスペ廃道";
  window.RSTH_IH._selectedMapId = 1;

  window.RSTH_IH._showSubHUD = false;

  // 初期選択キャラクターid
  window.RSTH_IH._selectedActorId = 1;


  // Npc定義用
  window.RSTH_IH.NpcD_mzonoff = window.RSTH_IH.parameters["NpcD_mzonoff"] === "true";

  const npcDefinitionsRaw = JSON.parse(window.RSTH_IH.parameters["NpcDefinitions"] || "[]");
  window.RSTH_IH.npcDefinitions = npcDefinitionsRaw.map(raw => {
    const obj = JSON.parse(raw);
    obj.isEnemy = obj.isEnemy === "true";
    obj.databaseId = Number(obj.databaseId);
    obj.defaultLv = Number(obj.defaultLv);
    obj.defaultExp = Number(obj.defaultExp);
    obj.moveSpd = Number(obj.moveSpd);

    if (window.RSTH_IH.NpcD_mzonoff) {
      // MZ形式(8キャラシート)
      //console.log("aaaa");
      obj.characterIndex = Number(obj.characterIndex || 0);
      //console.log("obj.characterIndex", obj.characterIndex);
      obj.faceIndex = Number(obj.faceIndex || 0);
      //console.log("obj.faceIndex", obj.faceIndex);
    } else {
      // 1キャラシート形式なので index は無視する
      //console.log("bbbb");
      obj.characterIndex = null;
      obj.faceIndex = null;
    }

    return obj;
  });

  window.RSTH_IH.NpcDefinitions = window.RSTH_IH.npcDefinitions;
  //console.log("window.RSTH_IH.NpcDefinitions", window.RSTH_IH.NpcDefinitions);
  window.RSTH_IH.NpcCount = 0;


  // mob定義用
  window.RSTH_IH.MobD_mzonoff = window.RSTH_IH.parameters["MobD_mzonoff"] === "true";

  const mobDefinitionsRaw = JSON.parse(window.RSTH_IH.parameters["MobDefinitions"] || "[]");
  window.RSTH_IH.mobDefinitions = mobDefinitionsRaw.map(raw => {
    const obj = JSON.parse(raw);
    obj.isEnemy = obj.isEnemy === "true";
    obj.databaseId = Number(obj.databaseId);
    obj.defaultLv = Number(obj.defaultLv);
    obj.defaultExp = Number(obj.defaultExp);
    obj.moveSpd = Number(obj.moveSpd);

    if (window.RSTH_IH.MobD_mzonoff) {
      // MZ形式(8キャラシート)
      obj.characterIndex = Number(obj.characterIndex || 0);
    } else {
      // 1キャラシート形式なので index は無視する
      obj.characterIndex = null;
    }

    return obj;
  });


  window.RSTH_IH.MobDefinitions = window.RSTH_IH.mobDefinitions;

  window.RSTH_IH.MobCount = 0;
  window.RSTH_IH.BossCount = 0;
  window.RSTH_IH.MaxMobAmount = 200;
  window.RSTH_IH.TreasureMobCount = 40;
  window.RSTH_IH.EnableAutoMobSpawn = false;
  window.RSTH_IH.AutoMobSpawnSet = 1;
  window.RSTH_IH.KilledMobCount = 0;

  window.RSTH_IH._weaponCooldown = 0;
  window.RSTH_IH._skillA_Cooldown = 0;
  window.RSTH_IH._skillB_Cooldown = 0;
  window.RSTH_IH._skillC_Cooldown = 0;

  window.RSTH_IH._cooldownMap = {};

  window.RSTH_IH.DmgPopup = window.RSTH_IH.parameters["DmgPopup"] === "true";

  window.RSTH_IH.TimeCount = 0;
  window.RSTH_IH._elapsedTimeEnabled = false;
  window.RSTH_IH._elapsedLimit = 300;
  window.RSTH_IH._progressGoalTime = 300; // 例：60秒（デフォルト）
  window.RSTH_IH.setProgressGoalTime = function (sec) {
    window.RSTH_IH._progressGoalTime = Math.max(1, sec); // 1秒未満は不可
  };

  window.RSTH_IH._timerNum = 0; // 表示するタイマーの決定 0で通常、1でボス用
  window.RSTH_IH._bossMob = null;

  window.RSTH_IH._questTextSprite = null;
  window.RSTH_IH._questTextBgSprite = null;
  window.RSTH_IH._questTextDuration = 0;
  window.RSTH_IH._questTextData = null;

  // 独自パラメータの定数ID（paramId 10以降を使用）
  window.RSTH_IH.PARAM_HIT = 10;
  window.RSTH_IH.PARAM_EVA = 11;
  window.RSTH_IH.PARAM_CRI = 12;
  window.RSTH_IH.PARAM_CRIEVA = 13;
  window.RSTH_IH.PARAM_ATKSPD = 14;

  window.RSTH_IH.shopCallFlag = 0;
  window.RSTH_IH.treasureCallFlag = 0;

  window.RSTH_IH.powerUpValueRates = {
    HP: 25,
    Str: 3,
    Vit: 3,
    Dex: 3,
    Agi: 3,
    Luk: 3,
    Drop: 2,
    Lv: 5,
    MaxLv: 5,
    Zeny: 5000,
  };

  window.RSTH_IH.LvupEasyflagFromPowerUP = 0;

  // 装備中のお供
  window.RSTH_IH.EquipmentOtomoIdActor1 = null;
  window.RSTH_IH.EquipmentOtomoIdActor2 = null;
  window.RSTH_IH.EquipmentOtomoIdActor3 = null;

  window.RSTH_IH.initPriorityVal = [1, 0, 4, 0, 1, 0];

  //アチーブメント用
  window.RSTH_IH = window.RSTH_IH || {};
  window.RSTH_IH._achievementBitmapCache = {};
  window.RSTH_IH._achievementImageNames = [
    "achv_icon",
    "reward_icon",
    "achv_done",
    "achv_notyet"
  ];

  window.RSTH_IH.preloadAchievementImages = function () {
    const cache = window.RSTH_IH._achievementBitmapCache;
    const names = window.RSTH_IH._achievementImageNames;
    names.forEach(name => {
      cache[name] = ImageManager.loadSystem(name);
    });
  };

  window.RSTH_IH.areAchievementImagesReady = function () {
    const names = window.RSTH_IH._achievementImageNames;
    return names.every(name => ImageManager.loadSystem(name).isReady());
  };

  window.RSTH_IH.unshoShopSymbols = [
    "HP", "Str", "Vit", "Dex", "Agi", "Luk", "Drop", "Lv", "MaxLv", "Zeny",
    "KB", "HS", "IG", "DFL", "ST1", "ST2", "RG", "RV",
  ];

  window.RSTH_IH.reviveFlag = false;

  window.RSTH_IH.achievementsSymbols = [
    "HP300", "HP600", "Str90", "Str200", "Vit90", "Vit200",
    "Dex90", "Dex200", "Agi90", "Agi200", "Luk90", "Luk200",
    "Lv99", "Lv150", "ULv99", "ULv150", "SLv99", "SLv150", "PLv99", "PLv150",
    "Boss1", "Boss2", "Boss3", "Boss4", "Boss5", "Boss5_5min",
    "Dmg200", "Dmg500", "FKL", "MKL1000", "MKL3000", "MKL5000",
  ];

  window.RSTH_IH.getAchievementsCheckBox = function () {
    return {
      FKL: {
        label: "★初めて敵を倒した！",
        conditions: "条件：初めて敵を倒す",
        value: 20,
        achieved: ConfigManager["achi_FKL"]
      },
      MKL1000: {
        label: "★敵を1000体倒した！",
        conditions: "条件：初めて敵を1000体倒す",
        value: 50,
        achieved: ConfigManager["achi_MKL1000"]
      },
      MKL3000: {
        label: "★敵を3000体倒した！",
        conditions: "条件：初めて敵を3000体倒す",
        value: 100,
        achieved: ConfigManager["achi_MKL3000"]
      },
      MKL5000: {
        label: "★敵を5000体倒した！",
        conditions: "条件：初めて敵を5000体倒す",
        value: 200,
        achieved: ConfigManager["achi_MKL5000"]
      },
      HP300: {
        label: "★HPが300を超えた",
        conditions: "条件：HPを300以上にする",
        value: 20,
        achieved: ConfigManager["achi_HP300"]
      },
      HP600: {
        label: "★HPが600を超えた",
        conditions: "条件：HPを600以上にする",
        value: 100,
        achieved: ConfigManager["achi_HP600"]
      },
      Str90: {
        label: "★Strが90を超えた",
        conditions: "条件：Strを90以上にする",
        value: 20,
        achieved: ConfigManager["achi_Str90"]
      },
      Str200: {
        label: "★Strが200を超えた",
        conditions: "条件：Strを200以上にする",
        value: 75,
        achieved: ConfigManager["achi_Str200"]
      },
      Vit90: {
        label: "★Vitが90を超えた",
        conditions: "条件：Vitを90以上にする",
        value: 20,
        achieved: ConfigManager["achi_Vit90"]
      },
      Vit200: {
        label: "★Vitが200を超えた",
        conditions: "条件：Vitを200以上にする",
        value: 75,
        achieved: ConfigManager["achi_Vit200"]
      },
      Dex90: {
        label: "★Dexが90を超えた",
        conditions: "条件：Dexを90以上にする",
        value: 20,
        achieved: ConfigManager["achi_Dex90"]
      },
      Dex200: {
        label: "★Dexが200を超えた",
        conditions: "条件：Dexを200以上にする",
        value: 75,
        achieved: ConfigManager["achi_Dex200"]
      },
      Agi90: {
        label: "★Agiが90を超えた",
        conditions: "条件：Agiを90以上にする",
        value: 20,
        achieved: ConfigManager["achi_Agi90"]
      },
      Agi200: {
        label: "★Agiが200を超えた",
        conditions: "条件：Agiを200以上にする",
        value: 75,
        achieved: ConfigManager["achi_Agi200"]
      },
      Luk90: {
        label: "★Lukが90を超えた",
        conditions: "条件：Lukを90以上にする",
        value: 20,
        achieved: ConfigManager["achi_Luk90"]
      },
      Luk200: {
        label: "★Lukが200を超えた",
        conditions: "条件：Lukを200以上にする",
        value: 75,
        achieved: ConfigManager["achi_Luk200"]
      },
      Lv99: {
        label: "★Lv99!",
        conditions: "条件：Lvを99にする",
        value: 20,
        achieved: ConfigManager["achi_Lv99"]
      },
      Lv150: {
        label: "★Lv150!!",
        conditions: "条件：Lvを150にする",
        value: 1000,
        achieved: ConfigManager["achi_Lv150"]
      },
      SLv99: {
        label: "★セリスLv99!",
        conditions: "条件：セリスのLvを99にする",
        value: 50,
        achieved: ConfigManager["achi_SLv99"]
      },
      SLv150: {
        label: "★セリスLv150!!",
        conditions: "条件：セリスのLvを150にする",
        value: 1000,
        achieved: ConfigManager["achi_SLv150"]
      },
      ULv99: {
        label: "★うゆゆLv99!",
        conditions: "条件：うゆゆのLvを99にする",
        value: 50,
        achieved: ConfigManager["achi_ULv99"]
      },
      ULv150: {
        label: "★うゆゆLv150!!",
        conditions: "条件：うゆゆのLvを150にする",
        value: 1000,
        achieved: ConfigManager["achi_ULv150"]
      },
      PLv99: {
        label: "★ぽまえらさんLv99!",
        conditions: "条件：ぽまえらさんのLvを99にする",
        value: 50,
        achieved: ConfigManager["achi_PLv99"]
      },
      PLv150: {
        label: "★ぽまえらさんLv150!!",
        conditions: "条件：ぽまえらさんのLvを150にする",
        value: 1000,
        achieved: ConfigManager["achi_PLv150"]
      },
      Boss1: {
        label: "★イテルスペ廃道のBOSS討伐",
        conditions: "条件：イテルスペ廃道のBOSSを討伐する",
        value: 20,
        achieved: ConfigManager["achi_Boss1"]
      },
      Boss2: {
        label: "★タルダモルス荒野のBOSS討伐",
        conditions: "条件：タルダモルス荒野のBOSSを討伐する",
        value: 50,
        achieved: ConfigManager["achi_Boss2"]
      },
      Boss3: {
        label: "★旧魔王城庭園のBOSS討伐",
        conditions: "条件：旧魔王城庭園のBOSSを討伐する",
        value: 150,
        achieved: ConfigManager["achi_Boss3"]
      },
      Boss4: {
        label: "★イグニス要塞前のBOSS討伐",
        conditions: "条件：イグニス要塞前のBOSSを討伐する",
        value: 500,
        achieved: ConfigManager["achi_Boss4"]
      },
      Boss5: {
        label: "★旧魔王の影討伐",
        conditions: "条件：旧魔王の影を討伐する",
        value: 1500,
        achieved: ConfigManager["achi_Boss5"]
      },
      Boss5_5min: {
        label: "★うゆゆで5分以内に旧魔王の影討伐",
        conditions: "条件：うゆゆで旧魔王の影を5分以内に討伐する",
        value: 5000,
        achieved: ConfigManager["achi_Boss5_5min"]
      },
      Dmg200: {
        label: "★与ダメージが200を超えた！",
        conditions: "条件：敵に1撃で200以上のダメージを与える",
        value: 30,
        achieved: ConfigManager["achi_Dmg200"]
      },
      Dmg500: {
        label: "★与ダメージが500を超えた！",
        conditions: "条件：敵に1撃で500以上のダメージを与える",
        value: 100,
        achieved: ConfigManager["achi_Dmg500"]
      },
    }
  };


  window.RSTH_IH.MobLankTable = {
    lank_1: [], lank_2: [], lank_3: [], lank_4: [], lank_5: [],
    lank_6: [], lank_7: [], lank_8: [], lank_9: []
  }

  // クエスト発生フラグ
  window.RSTH_IH.questSpawned = false;

  // ConfigManager に保存対象を追加
  ConfigManager.mapBgmName = "";
  ConfigManager.lvUpOptionMode = "nml";
  ConfigManager.treasureEffectMode = "nml";
  ConfigManager.priorityValues = window.RSTH_IH.initPriorityVal;
  ConfigManager.autoPotionPercent = 75;
  ConfigManager.commandRemember = false;
  ConfigManager.touchUI = false;
  ConfigManager.unshoP = Number(window.RSTH_IH.unshoP) || 0;
  ConfigManager.OtomoList = [];

  ConfigManager.alwaysDash = "on";

  ConfigManager.bgmVolume = 50;
  ConfigManager.seVolume = 50;

  ConfigManager.bgsVolume = 50;
  ConfigManager.meVolume = 50;

  // 初期装備類
  const types = ["Weapon", "Armor", "Skill", "Item", "Otomo"];
  const counts = { Weapon: 4, Armor: 4, Skill: 3, Item: 3, Otomo: 1 };

  for (const type of types) {
    const max = counts[type];
    for (let i = 1; i <= max; i++) {
      ConfigManager[`Actor1_${type}${i}`] = [];
      ConfigManager[`Actor2_${type}${i}`] = [];
      ConfigManager[`Actor3_${type}${i}`] = [];
    }
  }

  ConfigManager.InitEquip = false;

  ConfigManager.InitEquipActor1Slots = false;
  ConfigManager.InitEquipActor2Slots = false;
  ConfigManager.InitEquipActor3Slots = false;



  ConfigManager.makeData = (function () {
    const base = ConfigManager.makeData;
    return function () {
      const config = base.call(this);
      config.lvUpOptionMode = this.lvUpOptionMode;
      config.treasureEffectMode = this.treasureEffectMode;
      config.priorityValues = this.priorityValues;
      config.mapBgmName = this.mapBgmName;
      config.autoPotionPercent = this.autoPotionPercent;
      config.unshoP = this.unshoP;

      //console.log("ConfigManager.makeData");
      // 🟦 追加：gauge_◯◯各種保存
      const gaugeSymbols = window.RSTH_IH.unshoShopSymbols;
      for (const symbol of gaugeSymbols) {
        config[`gauge_${symbol}`] = this[`gauge_${symbol}`] ?? 0;
        //console.log("config[`gauge_${symbol}`]", config[`gauge_${symbol}`]);
      }

      const achievementsSymbols = window.RSTH_IH.achievementsSymbols;
      for (const symbol of achievementsSymbols) {
        config[`achi_${symbol}`] = this[`achi_${symbol}`] ?? false;
        //console.log("config[`achi_${symbol}`]", config[`achi_${symbol}`]);
      }


      // 初期装備類
      const types = ["Weapon", "Armor", "Skill", "Item", "Otomo"];
      const counts = { Weapon: 4, Armor: 4, Skill: 3, Item: 3, Otomo: 1 };

      for (const type of types) {
        const max = counts[type];
        for (let i = 1; i <= max; i++) {
          const key1 = `Actor1_${type}${i}`;
          const key2 = `Actor2_${type}${i}`;
          const key3 = `Actor3_${type}${i}`;
          config[key1] = this[key1];
          config[key2] = this[key2];
          config[key3] = this[key3];
        }
      }

      config.InitEquip = this.InitEquip;


      config.InitEquipActor1Slots = this.InitEquipActor1Slots;
      config.InitEquipActor2Slots = this.InitEquipActor2Slots;
      config.InitEquipActor3Slots = this.InitEquipActor3Slots;

      config.OtomoList = this.OtomoList;

      return config;
    };
  })();


  ConfigManager.applyData = (function () {
    const base = ConfigManager.applyData;
    return function (config) {
      base.call(this, config);

      this.commandRemember = this.readFlag(config, "commandRemember", false);
      this.touchUI = this.readFlag(config, "touchUI", false);
      this.lvUpOptionMode = config.lvUpOptionMode ?? "nml";
      this.treasureEffectMode = config.treasureEffectMode ?? "nml";
      this.autoPotionPercent = config.autoPotionPercent;
      this.mapBgmName = config.mapBgmName ?? "";
      this.unshoP = Number(config.unshoP) || 0;

      if (Array.isArray(config.priorityValues) && config.priorityValues.length === 6) {
        this.priorityValues = config.priorityValues;
      } else {
        this.priorityValues = window.RSTH_IH.initPriorityVal;
      }

      //console.log("ConfigManager.applyData");
      // 🟩 追加：gauge_◯◯各種読込
      const gaugeSymbols = window.RSTH_IH.unshoShopSymbols;
      for (const symbol of gaugeSymbols) {
        this[`gauge_${symbol}`] = Number(config[`gauge_${symbol}`]) || 0;
        //console.log("this[`gauge_${symbol}`]", this[`gauge_${symbol}`]);
      }

      const achievementsSymbols = window.RSTH_IH.achievementsSymbols;
      for (const symbol of achievementsSymbols) {
        this[`achi_${symbol}`] = config[`achi_${symbol}`] || false;
        //console.log("this[`achi_${symbol}`]", this[`achi_${symbol}`]);
      }


      // 初期装備類
      const types = ["Weapon", "Armor", "Skill", "Item", "Otomo"];
      const counts = { Weapon: 4, Armor: 4, Skill: 3, Item: 3, Otomo: 1 };

      for (const type of types) {
        const max = counts[type];
        for (let i = 1; i <= max; i++) {
          const key1 = `Actor1_${type}${i}`;
          const key2 = `Actor2_${type}${i}`;
          const key3 = `Actor3_${type}${i}`;
          this[key1] = config[key1];
          this[key2] = config[key2];
          this[key3] = config[key3];
        }
      }

      this.InitEquip = config.InitEquip ?? false;

      this.InitEquipActor1Slots = config.InitEquipActor1Slots || false;
      this.InitEquipActor2Slots = config.InitEquipActor2Slots || false;
      this.InitEquipActor3Slots = config.InitEquipActor3Slots || false;

      this.OtomoList = config.OtomoList || [];
    };
  })();



  // オートタイルshape画像取得用
  const paramList = JSON.parse(window.RSTH_IH.parameters["AutotileShapeList"] || "[]");
  window.RSTH_IH.AutotileShapeConfig = paramList.map(str => {
    const obj = JSON.parse(str);
    return {
      tilesetName: obj.TilesetName
    };
  });

  window.RSTH_IH.AutotileShapeBitmaskCache = { data: {} };


  window.RSTH_IH.AutotileBitmapCache = {};


  window.RSTH_IH.ItemTypeCache = {};  // itemId → { tileType, blockType } キャッシュ


  const _DataManager_loadDatabase = DataManager.loadDatabase;
  DataManager.loadDatabase = function () {
    _DataManager_loadDatabase.call(this);
    this.loadDataFile('FloorAutotileCache', 'FloorAutotileCache.json');
    this.loadDataFile('WallAutotileCache', 'WallAutotileCache.json');
  };


  const _DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
  DataManager.isDatabaseLoaded = function () {
    if (RSTH_DEBUG_LOG) console.warn("[isDatabaseLoaded] start");
    if (!_DataManager_isDatabaseLoaded.call(this)) return false;


    if (!this._autotileCacheInitialized) {
      this._autotileCacheInitialized = true;

      // --- JSONキャッシュ読み込み ---
      const floorObj = window.FloorAutotileCache;
      if (floorObj) {
        window.RSTH_IH.FloorAutotileBitmaskToShape = new Map(
          Object.entries(floorObj).map(([k, v]) => [Number(k), v])
        );
      }

      const wallObj = window.WallAutotileCache;
      if (wallObj) {
        window.RSTH_IH.WallAutotileBitmaskToShape = new Map(
          Object.entries(wallObj).map(([k, v]) => [Number(k), v])
        );
      }


      if (RSTH_DEBUG_LOG) console.warn("floorObj ", floorObj);
      if (RSTH_DEBUG_LOG) console.warn("wallObj ", wallObj);
      // --- Autotile画像キャッシュ読み込み ---
      window.RSTH_IH.loadAutotileBitmapCacheImages();


      // item.meta.tileTypeとblockTypeをキャッシュ化
      for (let itemId = 1; itemId < $dataItems.length; itemId++) {
        const item = $dataItems[itemId];
        if (!item || !item.meta) continue;
        const tileType = Number(item.meta.tileType || 0);
        const blockType = item.meta.blockType || "ground";
        const tileset = item.meta.tileset || "Outside_A2";
        window.RSTH_IH.ItemTypeCache[itemId] = { tileType, blockType, tileset };
      }
      if (RSTH_DEBUG_LOG) console.log("[Sprite_AutotileBlock][initialize] window.RSTH_IH.ItemTypeCache", window.RSTH_IH.ItemTypeCache);
    }

    return true;
  };

  window.RSTH_IH.loadAutotileBitmapCacheImages = function () {
    if (RSTH_DEBUG_LOG) console.warn("[loadAutotileBitmapCacheImages] start");

    if (!window.RSTH_IH.AutotileBitmapCache) window.RSTH_IH.AutotileBitmapCache = {};

    const paramListRaw = window.RSTH_IH.parameters["AutotileShapeList"] || "[]";
    const paramList = JSON.parse(paramListRaw);

    if (RSTH_DEBUG_LOG) console.warn("[loadAutotileBitmapCacheImages] paramList", paramList);

    for (const jsonStr of paramList) {
      let entry;
      try {
        entry = JSON.parse(jsonStr);
      } catch (e) {
        if (RSTH_DEBUG_LOG) console.warn("JSONパース失敗:", jsonStr);
        continue;
      }

      const filename = entry.TilesetName;

      const match = filename.match(/(floor|wall)_tileId(\d+)_(.+)/i);
      if (!match) {
        if (RSTH_DEBUG_LOG) console.warn("ファイル名パース失敗:", filename);
        continue;
      }

      const typeName = match[1];
      const tileId = parseInt(match[2]);
      const tilesetName = match[3];

      const path = `img/tilesets/${filename}`;
      const bitmap = ImageManager.loadBitmap("", path);



      if (!window.RSTH_IH.AutotileBitmapCache[tilesetName]) {
        window.RSTH_IH.AutotileBitmapCache[tilesetName] = {};
      }
      if (!window.RSTH_IH.AutotileBitmapCache[tilesetName][tileId]) {
        window.RSTH_IH.AutotileBitmapCache[tilesetName][tileId] = {};
      }
      window.RSTH_IH.AutotileBitmapCache[tilesetName][tileId][typeName] = bitmap;

      if (RSTH_DEBUG_LOG) console.warn("登録:", { tilesetName, tileId, typeName, filename });
    }
  };




  window.RSTH_IH.FloorAutotileBitmaskToShape = new Map([
    [0, 46],
    [1, 45],
    [2, 46],
    [3, 45],
    [4, 42],
    [5, 37],
    [6, 42],
    [7, 36],
    [8, 46],
    [9, 45],
    [10, 46],
    [11, 45],
    [12, 42],
    [13, 37],
    [14, 42],
    [15, 36],
    [16, 43],
    [17, 33],
    [18, 43],
    [19, 33],
    [20, 35],
    [21, 23],
    [22, 35],
    [23, 21],
    [24, 43],
    [25, 33],
    [26, 43],
    [27, 33],
    [28, 34],
    [29, 22],
    [30, 34],
    [31, 20],
    [32, 46],
    [33, 45],
    [34, 46],
    [35, 45],
    [36, 42],
    [37, 37],
    [38, 42],
    [39, 36],
    [40, 46],
    [41, 45],
    [42, 46],
    [43, 45],
    [44, 42],
    [45, 37],
    [46, 42],
    [47, 36],
    [48, 43],
    [49, 33],
    [50, 43],
    [51, 33],
    [52, 35],
    [53, 23],
    [54, 35],
    [55, 21],
    [56, 43],
    [57, 33],
    [58, 43],
    [59, 33],
    [60, 34],
    [61, 22],
    [62, 34],
    [63, 20],
    [64, 44],
    [65, 39],
    [66, 44],
    [67, 39],
    [68, 32],
    [69, 27],
    [70, 32],
    [71, 26],
    [72, 44],
    [73, 39],
    [74, 44],
    [75, 39],
    [76, 32],
    [77, 27],
    [78, 32],
    [79, 26],
    [80, 41],
    [81, 31],
    [82, 41],
    [83, 31],
    [84, 19],
    [85, 15],
    [86, 19],
    [87, 7],
    [88, 41],
    [89, 31],
    [90, 41],
    [91, 31],
    [92, 17],
    [93, 11],
    [94, 17],
    [95, 3],
    [96, 44],
    [97, 39],
    [98, 44],
    [99, 39],
    [100, 32],
    [101, 27],
    [102, 32],
    [103, 26],
    [104, 44],
    [105, 39],
    [106, 44],
    [107, 39],
    [108, 32],
    [109, 27],
    [110, 32],
    [111, 26],
    [112, 40],
    [113, 29],
    [114, 40],
    [115, 29],
    [116, 18],
    [117, 13],
    [118, 18],
    [119, 5],
    [120, 40],
    [121, 29],
    [122, 40],
    [123, 29],
    [124, 16],
    [125, 9],
    [126, 16],
    [127, 1],
    [128, 46],
    [129, 45],
    [130, 46],
    [131, 45],
    [132, 42],
    [133, 37],
    [134, 42],
    [135, 36],
    [136, 46],
    [137, 45],
    [138, 46],
    [139, 45],
    [140, 42],
    [141, 37],
    [142, 42],
    [143, 36],
    [144, 43],
    [145, 33],
    [146, 43],
    [147, 33],
    [148, 35],
    [149, 23],
    [150, 35],
    [151, 21],
    [152, 43],
    [153, 33],
    [154, 43],
    [155, 33],
    [156, 34],
    [157, 22],
    [158, 34],
    [159, 20],
    [160, 46],
    [161, 45],
    [162, 46],
    [163, 45],
    [164, 42],
    [165, 37],
    [166, 42],
    [167, 36],
    [168, 46],
    [169, 45],
    [170, 46],
    [171, 45],
    [172, 42],
    [173, 37],
    [174, 42],
    [175, 36],
    [176, 43],
    [177, 33],
    [178, 43],
    [179, 33],
    [180, 35],
    [181, 23],
    [182, 35],
    [183, 21],
    [184, 43],
    [185, 33],
    [186, 43],
    [187, 33],
    [188, 34],
    [189, 22],
    [190, 34],
    [191, 20],
    [192, 44],
    [193, 38],
    [194, 44],
    [195, 38],
    [196, 32],
    [197, 25],
    [198, 32],
    [199, 24],
    [200, 44],
    [201, 38],
    [202, 44],
    [203, 38],
    [204, 32],
    [205, 25],
    [206, 32],
    [207, 24],
    [208, 41],
    [209, 30],
    [210, 41],
    [211, 30],
    [212, 19],
    [213, 14],
    [214, 19],
    [215, 6],
    [216, 41],
    [217, 30],
    [218, 41],
    [219, 30],
    [220, 17],
    [221, 10],
    [222, 17],
    [223, 2],
    [224, 44],
    [225, 38],
    [226, 44],
    [227, 38],
    [228, 32],
    [229, 25],
    [230, 32],
    [231, 24],
    [232, 44],
    [233, 38],
    [234, 44],
    [235, 38],
    [236, 32],
    [237, 25],
    [238, 32],
    [239, 24],
    [240, 40],
    [241, 28],
    [242, 40],
    [243, 28],
    [244, 18],
    [245, 12],
    [246, 18],
    [247, 4],
    [248, 40],
    [249, 28],
    [250, 40],
    [251, 28],
    [252, 16],
    [253, 8],
    [254, 16],
    [255, 0]
  ]);

  window.RSTH_IH.WallAutotileBitmaskToShape = new Map([
    [0, 15],  // 単体15
    [1, 14],
    [2, 15],
    [3, 15],
    [4, 7],
    [5, 7],
    [6, 7],
    [7, 6],
    [8, 15],
    [9, 14],
    [10, 15],
    [11, 15],
    [12, 7],
    [13, 7],
    [14, 7],
    [15, 6],
    [16, 11],
    [17, 10],
    [18, 11],
    [19, 10],
    [20, 7],
    [21, 7],
    [22, 7],
    [23, 6],
    [24, 15],
    [25, 14],
    [26, 15],
    [27, 15],
    [28, 3],
    [29, 3],
    [30, 3],
    [31, 2],
    [32, 15],
    [33, 14],
    [34, 15],
    [35, 15],
    [36, 7],
    [37, 7],
    [38, 7],
    [39, 6],
    [40, 15],
    [41, 14],
    [42, 15],
    [43, 15],
    [44, 7],
    [45, 7],
    [46, 7],
    [47, 6],
    [48, 15],
    [49, 14],
    [50, 15],
    [51, 15],
    [52, 7],
    [53, 7],
    [54, 7],
    [55, 6],
    [56, 15],
    [57, 14],
    [58, 15],
    [59, 15],
    [60, 7],
    [61, 7],
    [62, 7],
    [63, 6],
    [64, 13],
    [65, 13],
    [66, 13],
    [67, 13],
    [68, 5],
    [69, 5],
    [70, 5],
    [71, 5],
    [72, 13],
    [73, 13],
    [74, 13],
    [75, 13],
    [76, 5],
    [77, 5],
    [78, 5],
    [79, 5],
    [80, 13],
    [81, 13],
    [82, 13],
    [83, 13],
    [84, 5],
    [85, 5],
    [86, 5],
    [87, 5],
    [88, 13],
    [89, 13],
    [90, 13],
    [91, 13],
    [92, 5],
    [93, 5],
    [94, 5],
    [95, 5],
    [96, 13],
    [97, 13],
    [98, 13],
    [99, 13],
    [100, 5],
    [101, 5],
    [102, 5],
    [103, 5],
    [104, 13],
    [105, 13],
    [106, 13],
    [107, 13],
    [108, 5],
    [109, 5],
    [110, 5],
    [111, 5],
    [112, 9],
    [113, 9],
    [114, 9],
    [115, 9],
    [116, 5],
    [117, 5],
    [118, 5],
    [119, 5],
    [120, 13],
    [121, 13],
    [122, 13],
    [123, 13],
    [124, 1],
    [125, 1],
    [126, 1],
    [127, 1],
    [128, 15],
    [129, 14],
    [130, 15],
    [131, 14],
    [132, 7],
    [133, 7],
    [134, 7],
    [135, 7],
    [136, 15],
    [137, 15],
    [138, 15],
    [139, 15],
    [140, 7],
    [141, 7],
    [142, 7],
    [143, 7],
    [144, 11],
    [145, 11],
    [146, 11],
    [147, 11],
    [148, 7],
    [149, 7],
    [150, 7],
    [151, 7],
    [152, 11],
    [153, 15],
    [154, 15],
    [155, 15],
    [156, 3],
    [157, 3],
    [158, 3],
    [159, 3],
    [160, 15],
    [161, 15],
    [162, 15],
    [163, 15],
    [164, 7],
    [165, 7],
    [166, 7],
    [167, 7],
    [168, 15],
    [169, 15],
    [170, 15],
    [171, 15],
    [172, 7],
    [173, 7],
    [174, 7],
    [175, 7],
    [176, 15],
    [177, 15],
    [178, 15],
    [179, 15],
    [180, 7],
    [181, 7],
    [182, 7],
    [183, 7],
    [184, 15],
    [185, 15],
    [186, 15],
    [187, 15],
    [188, 7],
    [189, 7],
    [190, 7],
    [191, 7],
    [192, 13],
    [193, 12],
    [194, 13],
    [195, 13],
    [196, 5],
    [197, 5],
    [198, 5],
    [199, 4],
    [200, 13],
    [201, 4],
    [202, 13],
    [203, 13],
    [204, 5],
    [205, 5],
    [206, 5],
    [207, 4],
    [208, 13],
    [209, 12],
    [210, 13],
    [211, 13],
    [212, 5],
    [213, 5],
    [214, 5],
    [215, 4],
    [216, 13],
    [217, 12],
    [218, 13],
    [219, 13],
    [220, 5],
    [221, 5],
    [222, 5],
    [223, 4],
    [224, 13],
    [225, 12],
    [226, 13],
    [227, 13],
    [228, 5],
    [229, 5],
    [230, 5],
    [231, 4],
    [232, 13],
    [233, 12],
    [234, 13],
    [235, 13],
    [236, 5],
    [237, 5],
    [238, 5],
    [239, 4],
    [240, 9],
    [241, 8],
    [242, 9],
    [243, 9],
    [244, 5],
    [245, 5],
    [246, 5],
    [247, 4],
    [248, 13],
    [249, 12],
    [250, 13],
    [251, 13],
    [252, 1],
    [253, 1],
    [254, 1],
    [255, 0]

  ]);



})();











